<?php
/**
 * Notification Triggers for LearnDash Slack
 *
 * @since		1.0.0
 *
 * @package LearnDash_Slack
 * @subpackage LearnDash_Slack/core/notifications
 */

defined( 'ABSPATH' ) || die();

final class LearnDash_Slack_Notification_Triggers {

	/**
	 * LearnDash_Slack_Notification_Triggers constructor.
	 * 
	 * @since		1.0.0
	 */
	function __construct() {

		// Triggers Pass/Fail/Completion of a Quiz
		add_action( 'learndash_quiz_completed', array( $this, 'quiz_completed' ), 10, 2 );
		
		// Triggers Completion of a Lesson
		add_action( 'learndash_lesson_completed', array( $this, 'lesson_completed' ) );
		
		// Triggers when Lesson becomes Available to a User
		add_action( 'learndash_update_course_access', array( $this, 'lesson_available' ), 10, 4 );
		
		// Triggers Completion of a Topic
		add_action( 'learndash_topic_completed', array( $this, 'topic_completed' ) );
		
		// Checks Daily for Expiring Courses
		add_action( 'ld_slack_daily_cron', array( $this, 'course_expires' ) );
		
		// Update User Meta to reflect Last Login Time
		add_action( 'wp_login', array( $this, 'update_last_login_time' ), 10, 2 );
		
		// Checks Daily for Users who have not logged in recently
		add_action( 'ld_slack_daily_cron', array( $this, 'user_has_not_logged_in_recently' ) );
		
		// Triggers when an Assignment is uploaded by a User
		add_action( 'learndash_assignment_uploaded', array( $this, 'upload_assignment' ), 10, 2 );
		
		// Triggers when an Assignment is approved by an Instructor/Admin
		add_action( 'learndash_assignment_approved', array( $this, 'approve_assignment' ) );
		
		// Triggers when an Essay is manually graded
		add_action( 'learndash_essay_all_quiz_data_updated', array( $this, 'essay_graded' ), 10, 4 );
		
		//Triggers when a User is enrolled in a Course
		add_action( 'learndash_update_course_access', array( $this, 'enroll_course' ), 10, 4 );
		
		// Triggers on Course Completion
		add_action( 'learndash_course_completed', array( $this, 'complete_course' ) );

	}

	/**
	 * Triggers on Quiz Completion, regardless of result
	 * 
	 * @param		array $quiz_data	Quiz data
	 * @param		object $current_user WP_User Object of the quiz taker
	 *														
	 * @access		public
	 * @since		1.0.0
	 * return		void
	 */
	public function quiz_completed( $quiz_data, $current_user ) {

		$quiz_args = array(
			'user_id' => $current_user->ID,
			'course_id' => ( $quiz_data['course']->ID ) ? $quiz_data['course']->ID : '',
			'quiz_id' => $quiz_data['quiz']->ID,
			'correct' => $quiz_data['score'],
			'incorrect' => $quiz_data['count'] - $quiz_data['score'],
			'total' => $quiz_data['count'],
			'percentage' => $quiz_data['percentage'],
			'timestamp' => $quiz_data['time'],
			'time_spent' => $quiz_data['timespent'],
			'graded' => $quiz_data['graded'],
		);

		// If user fails the quiz
		if ( $quiz_data['pass'] == 0 ) {

			do_action( 'ld_slack_notify', 'fail_quiz', $quiz_args );

		}
		
		// If user passes the quiz
		if ( $quiz_data['pass'] == 1 ) {
			
			do_action( 'ld_slack_notify', 'pass_quiz', $quiz_args );
			
		}
		
		// Fires on completion no matter the result
		do_action( 'ld_slack_notify', 'complete_quiz', $quiz_args );

	}
	
	/**
	 * Triggers on Lesson Completion
	 * 
	 * @param		array $lesson_data Lesson Data
	 *								  
	 * @access		public
	 * @since		1.0.0
	 * @return		void
	 */
	public function lesson_completed( $lesson_data ) {

		do_action( 'ld_slack_notify', 'complete_lesson', array(
			'user_id' => $lesson_data['user']->ID,
			'course_id' => $lesson_data['course']->ID,
			'lesson_id' => $lesson_data['lesson']->ID,
			'timestamp' => time(), // %timestamp%
		) );
		
	}
	
	/**
	 * Triggers when a scheduled Lesson is available to a User
	 * This applies to "Make Lesson visible on specific date" Field, not the Post Scheduling Field
	 *
	 * @param		int   $user_id		ID of user who enroll
	 * @param		int   $course_id	ID of course enrolled into
	 * @param		array $access_list	List of users who have access to the course
	 * @param		bool  $remove		True if remove user access from a course | false otherwise
	 *																				  
	 * @access		public
	 * @since		1.0.0
	 * @return		void
	 */
	public function lesson_available( $user_id, $course_id, $access_list, $remove ) {

		// Exit if user removed from a course
		if ( $remove === true ) return;

		$lessons = learndash_get_lesson_list( $course_id );
		foreach ( $lessons as $lesson ) {
			
			$lesson_access_from = ld_lesson_access_from( $lesson->ID, $user_id );
			if ( ! is_null( $lesson_access_from ) ) {
				
				do_action( 'ld_slack_notify', 'lesson_available', array(
					'user_id' => $user_id,
					'course_id' => $course_id,
					'lesson_id' => $lesson->ID,
				) );
				
			}
			
		}
		
	}
	
	/**
	 * Triggers when a Topic is Completed
	 * 
	 * @param		array $topic_data Topic Data
	 *								
	 * @access		public
	 * @since		1.0.0
	 * @return		void
	 */
	public function topic_completed( $topic_data ) {
		
		do_action( 'ld_slack_notify', 'complete_topic', array(
			'user_id' => $topic_data['user']->ID,
			'course_id' => $topic_data['course']->ID,
			'lesson_id' => $topic_data['lesson']->ID,
			'topic_id' => $topic_data['topic']->ID,
			'timestamp' => time(), // %timestamp%
		) );
		
	}
	
	/**
	 * Send Course IDs for all Courses with Expiration Date that hasn't passed yet
	 * We'll determine which are X Days Away once we're processing the Notification
	 * 
	 * @access		public
	 * @since		1.0.0
	 * @return		void
	 */
	public function course_expires() {
		
		// Get all courses
		$args = array(
			'post_type'   => 'sfwd-courses',
			'post_status' => 'publish',
			'posts_per_page' => -1,
		);
		
		$courses = get_posts( $args );
		
		foreach ( $courses as $course ) {
			
			$course_meta = learndash_get_setting( $course->ID );
			$course_meta = maybe_unserialize( $course_meta );
			
			// If course doesn't have expiration settings, continue
			if ( ( ! isset( $course_meta['expire_access'] ) || ! isset( $course_meta['expire_access_days'] ) ) || 
				( isset( $course_meta['expire_access'] ) && $course_meta['expire_access'] !== 'on' ) || 
				( isset( $course_meta['expire_access_days'] ) && $course_meta['expire_access_days'] == 0 ) ) {
				continue;
			}
			
			$course_access_list = $course_meta['course_access_list'];
			
			// If course has no access list, continue
			if ( empty( $course_access_list ) ) continue; 
			
			$course_access_list = explode( ',', trim( $course_access_list ) );
			$course_access_days = (int) $course_meta['expire_access_days'];
			
			// We need to only check for one User, since we're only sending the notification once
			$user_id = trim( $course_access_list[0] );
			
			$access_from = get_user_meta( $user_id, 'course_' . $course->ID . '_access_from', true );
			
			$course_expire_time = strtotime( '+' . $course_access_days . ' days', $access_from );
			
			$course_notification_args = array(
				'course_id' => $course->ID,
				'expire_time' => $course_expire_time,
			);
			
			// If we're on LD >= 2.4, Course Points exist
			if ( version_compare( LEARNDASH_VERSION, '2.4' ) >= 0 ) {

				// Always report this even if Course Points for this Course is disabled
				$course_notification_args['student_course_points'] = learndash_get_user_course_points( $user_id );

				if ( learndash_get_course_points_enabled( $course_data['course']->ID ) ) {
					$course_notification_args['awarded_course_points'] = learndash_get_course_points( $course->ID );
					$course_notification_args['access_course_points'] = learndash_get_course_points_access( $course->ID );
				}
				else {

					// Course Points Disabled Message
					$course_points_disabled = sprintf( _x( '%s Points are not Enabled for this %s', 'Course Points are not Enabled for this Course', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'course' ), LearnDash_Custom_Label::get_label( 'course' ) );

					$course_notification_args['awarded_course_points'] = $course_points_disabled;
					$course_notification_args['access_course_points'] = $course_points_disabled;

				}

			}
			
			do_action( 'ld_slack_notify', 'course_expires', $course_notification_args );
			
		}
		
	}
	
	/**
	 * Update User Meta to reflect the last logged-in time
	 *
	 * @param		string $user_login Username
	 * @param		object $user	   WP_User Object
	 *										 
	 * @access		public
	 * @since		1.0.0
	 * @return		void
	 */
	public function update_last_login_time( $user_login, $user ) {
		update_user_meta( $user->ID, 'ld_slack_last_login', time() );
	}
	
	/**
	 * Send User IDs and Unix Epoch Timestamps for all Users every Day
	 * We'll determine who haven't logged in within X Days Away once we're processing the Notification
	 * 
	 * @access		public
	 * @since		1.0.0
	 * @return		void
	 */
	public function user_has_not_logged_in_recently() {
		
		$users = get_users( array(
			'role__in' => array( 'subscriber' ),
		) );

		foreach ( $users as $user ) {
			
			$last_login = get_user_meta( $user->ID, 'ld_slack_last_login', true );
			
			if ( empty( $last_login ) || ! isset( $last_login ) ) continue;

			do_action( 'ld_slack_notify', 'not_logged_in', array(
				'user_id' => $user->ID,
				'last_login' => $last_login,
			) );
			
		}
		
	}
	
	/**
	 * Triggers when a User Uploads an Assignment
	 * 
	 * @param		integer $assignment_id   Assignment Post ID
	 * @param		array   $assignment_meta Assignment Meta
	 *												  
	 * @access		public
	 * @since		1.0.0
	 * @return		void
	 */
	public function upload_assignment( $assignment_id, $assignment_meta ) {
		
		$user_id   = get_post_meta( $assignment_id, 'user_id', true );
		
		do_action( 'ld_slack_notify', 'upload_assignment', array(
			'user_id' => $user_id,
			'course_id' => $assignment_meta['course_id'],
			'lesson_id' => $assignment_meta['lesson_id'],
			'assignment_id' => $assignment_id,
		) );
		
	}
	
	/**
	 * Triggers when an Assignment is approved by an Instructor/Admin
	 * 
	 * @param		integer $assignment_id Assignment Post ID
	 *													 
	 * @access		public
	 * @since		1.0.0
	 * @return		void
	 */
	public function approve_assignment( $assignment_id ) {
		
		$user_id   = get_post_meta( $assignment_id, 'user_id', true );
		$course_id = learndash_get_course_id( $assignment_id );
		$lesson_id = get_post_meta( $assignment_id, 'lesson_id', true );
		
		do_action( 'ld_slack_notify', 'approve_assignment', array(
			'user_id' => $user_id,
			'course_id' => $course_id,
			'lesson_id' => $lesson_id,
			'assignment_id' => $assignment_id,
		) );
		
	}
	
	/**
	 * Triggers when an Essay Question is manually Graded
	 * 
	 * @param		integer $quiz_id			Quiz ID
	 * @param		integer $question_id		Question ID
	 * @param		object  $updated_scoring	Essay Object
	 * @param		object  $essay				Submitted Essay Object
	 *													   
	 * @access		public
	 * @since		1.0.0
	 * @return		void
	 */
	public function essay_graded( $quiz_id, $question_id, $updated_scoring, $essay ) {
		
		if ( $essay->post_status == 'graded' ) {
			
			$user_id = $essay->post_author;
			$real_quiz_id = learndash_get_quiz_id_by_pro_quiz_id( $quiz_id );		
			$course_id = learndash_get_course_id( $real_quiz_id );
			$lesson_id = learndash_get_lesson_id( $real_quiz_id );
			
			$question_mapper = new WpProQuiz_Model_QuestionMapper();
			$question_map = $question_mapper->fetchById( intval( $question_id ) );

			$quiz_data = get_user_meta( $user_id, '_sfwd-quizzes', true );

			$points_earned = '';
			foreach ( $quiz_data as $question ) {

				if ( ! is_null( $question['graded'] ) ) {

					foreach ( $question['graded'] as $question_key => $question_data ) {

						if ( $question_key == $question_id ) {
							$points_earned = $question_data['points_awarded'];
							continue 2;
						}

					}

				}

			}
			
			$points_possible = $question_map->getPoints();
			
			do_action( 'ld_slack_notify', 'essay_graded', array(
				'user_id' => $user_id,
				'course_id' => $course_id,
				'quiz_id' => $real_quiz_id,
				'points_earned' => $points_earned,
				'points_possible' => $points_possible,
			) );
			
		}
		
	}
	
	/**
	 * Triggers when a User Enrolls in a Course
	 * 
	 * @param		integer $user_id		User ID
	 * @param		integer $course_id		Course ID
	 * @param		array   $access_list	List of Users who have access to the Course
	 * @param		boolean $remove			True if remove user access from a course | false otherwise
	 *																					
	 * @access		public
	 * @since		1.0.0
	 * @return		void
	 */
	public function enroll_course( $user_id, $course_id, $access_list, $remove ) {
		
		if ( $remove === true ) return false;
		
		$course_notification_args = array(
			'user_id' => $user_id,
			'course_id' => $course_id,
		);
		
		// If we're on LD >= 2.4, Course Points exist
		if ( version_compare( LEARNDASH_VERSION, '2.4' ) >= 0 ) {
			
			// Always report this even if Course Points for this Course is disabled
			$course_notification_args['student_course_points'] = learndash_get_user_course_points( $user_id );
			
			if ( learndash_get_course_points_enabled( $course_id ) ) {
				$course_notification_args['access_course_points'] = learndash_get_course_points_access( $course_id );
			}
			else {
				
				// Course Points Disabled Message
				$course_points_disabled = sprintf( _x( '%s Points are not Enabled for this %s', 'Course Points are not Enabled for this Course', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'course' ), LearnDash_Custom_Label::get_label( 'course' ) );
				
				$course_notification_args['access_course_points'] = $course_points_disabled;
				
			}
			
		}
		
		do_action( 'ld_slack_notify', 'enroll_course', $course_notification_args );
		
	}
	
	/**
	 * Triggers on Course Completion
	 * 
	 * @param		array $course_data Course Data Array from LearnDash
	 *                                                   
	 * @access		public
	 * @since		1.0.0
	 * @return		void
	 */
	public function complete_course( $course_data ) {

		$timestamp = get_user_meta( $course_data['user']->ID, 'course_completed_' . $course_data['course']->ID, true );

		if ( empty( $timestamp ) ) {
			return '-';
		}
		
		$quiz_data = get_user_meta( $course_data['user']->ID, '_sfwd-quizzes', true );	
		global $wpdb;

		$quizzes = $wpdb->get_col( $wpdb->prepare( 'SELECT post_id FROM ' . $wpdb->postmeta . " WHERE meta_key = 'course_id' AND meta_value = '%d'", $course_data['course']->ID ) );

		if ( empty( $quizzes ) ) {
			$result = 0;
			return;
		}

		$scores = array();
		
		// Their Key => Our Key
		$fields = array(
			'score' => 'correct',
			'percentage' => 'percentage',
			'timespent' => 'time_spent',
			'count' => 'total',
		);
		
		// This sounds like a job for... Variable Variables!
		foreach ( $fields as $key => $new_key ) {

			if ( ( ! empty( $quiz_data ) ) && ( is_array( $quiz_data ) ) ) {

				foreach ( $quiz_data as $data ) {

					if ( in_array( $data['quiz'], $quizzes ) ) {

						if ( empty( $scores[ $data['quiz'] ] ) || $scores[ $data['quiz'] ] < $data[ $key ] ) {

							${"scores_$new_key"}[ $data['quiz'] ] = $data[ $key ];

						}

					}

				}

			}
			
			${"sum_$new_key"} = 0;
			foreach( ${"scores_$new_key"} as $score ) {
				${"sum_$new_key"} += $score;
			}
			
			// Average
			${"cumulative_$new_key"} = number_format( ${"sum_$new_key"} / count( ${"scores_$new_key"} ), 2 );
			
			// Sum
			${"aggregate_$new_key"} = number_format( ${"sum_$new_key"}, 2 );
			
		}
		
		$cumulative_time_spent = learndash_seconds_to_time( $cumulative_time_spent );
		$aggregate_time_spent = learndash_seconds_to_time( $aggregate_time_spent );
		
		$course_notification_args = array(
			'user_id' => $course_data['user']->ID,
			'course_id' => $course_data['course']->ID,
			'timestamp' => $timestamp,
			'cumulative_correct' => $cumulative_correct,
			'cumulative_incorrect' => $cumulative_total - $cumulative_correct,
			'cumulative_total' => $cumulative_total,
			'cumulative_percentage' => $cumulative_percentage,
			'cumulative_timespent' => $cumulative_time_spent,
			'aggregate_correct' => $aggregate_correct,
			'aggregate_incorrect' => $aggregate_total - $aggregate_correct,
			'aggregate_total' => $aggregate_total,
			'aggregate_percentage' => $aggregate_percentage,
			'aggregate_timespent' => $aggregate_time_spent,
		);
		
		// If we're on LD >= 2.4, Course Points exist
		if ( version_compare( LEARNDASH_VERSION, '2.4' ) >= 0 ) {
			
			// Always report this even if Course Points for this Course is disabled
			$course_notification_args['student_course_points'] = learndash_get_user_course_points( $course_data['user']->ID );
			
			if ( learndash_get_course_points_enabled( $course_data['course']->ID ) ) {
				$course_notification_args['awarded_course_points'] = learndash_get_course_points( $course_data['course']->ID );
				$course_notification_args['access_course_points'] = learndash_get_course_points_access( $course_data['course']->ID );
			}
			else {
				
				// Course Points Disabled Message
				$course_points_disabled = sprintf( _x( '%s Points are not Enabled for this %s', 'Course Points are not Enabled for this Course', 'learndash-slack' ), LearnDash_Custom_Label::get_label( 'course' ), LearnDash_Custom_Label::get_label( 'course' ) );
				
				$course_notification_args['awarded_course_points'] = $course_points_disabled;
				$course_notification_args['access_course_points'] = $course_points_disabled;
				
			}
			
		}
			
		do_action( 'ld_slack_notify', 'complete_course', $course_notification_args );
		
	}

}