<?php
/**
 * Notification Triggers get bounced here to be handled and pushed to Slack
 *
 * @since		1.0.0
 *
 * @package LearnDash_Slack
 * @subpackage LearnDash_Slack/core/notifications
 */

defined( 'ABSPATH' ) || die();

final class LearnDash_Slack_Notification_Handler {

	/**
	 * LearnDash_Slack_Notification_Handler constructor.
	 * 
	 * @since		1.0.0
	 */
	function __construct() {
		
		// Create Necessary Post Type(s) and set up Create/Update/Delete functionality
		add_action( 'init', array( $this, 'setup_notifications' ) );
		
		// Generic Notification Router
		add_action( 'ld_slack_notify', array( $this, 'notify' ), 10, 2 );

	}
	
	/**
	 * Creates Post Type(s) and necessary Create/Update/Delete functionality for Post Meta
	 * Controlled by the ld_slack_notifications Filter
	 * 
	 * @access		public
	 * @since		1.0.0
	 * @return		void
	 */
	public function setup_notifications() {
		
		global $ld_slack_notifications;
		
		$ld_slack_notifications = apply_filters( 'ld_slack_notifications', array() );
		
		if ( ! $ld_slack_notifications ) {
			return;
		}
		
		foreach ( $ld_slack_notifications as $notification_id => $notification_args ) {
		
			// Init the Post Types
			$this->create_notification_post_types( $notification_id, $notification_args );
			
			// Init Taxonomies to hold Webhooks
			$this->create_webhook_taxonomies( $notification_id, $notification_args );
			
		}
		
	}
	
	/**
	 * Handles Post Type(s) Creation
	 * 
	 * @param		string $notification_id   ID Used for Notification Hooks
	 * @param		array  $notification_args Array holding some basic Strings, but more importantly the Fields Array
	 *																										  
	 * @access		private
	 * @since		1.0.0
	 * @return		void
	 */
	private function create_notification_post_types( $notification_id, $notification_args ) {
		
		$labels = array(
			'name'			   => "$notification_args[name] Feeds",
			'singular_name'	  => "$notification_args[name] Feed",
			'menu_name'		  => "$notification_args[name] Feeds",
			'name_admin_bar'	 => "$notification_args[name] Feed",
			'add_new'			=> "Add New",
			'add_new_item'	   => "Add New $notification_args[name] Feed",
			'new_item'		   => "New $notification_args[name] Feed",
			'edit_item'		  => "Edit $notification_args[name] Feed",
			'view_item'		  => "View $notification_args[name] Feed",
			'all_items'		  => "All $notification_args[name] Feeds",
			'search_items'	   => "Search $notification_args[name] Feeds",
			'parent_item_colon'  => "Parent $notification_args[name] Feeds:",
			'not_found'		  => "No $notification_args[name] Feeds found.",
			'not_found_in_trash' => "No $notification_args[name] Feeds found in Trash.",
		);
		
		$args = array(
			'labels'			 => $labels,
			'public'			 => false,
			'publicly_queryable' => true,
			'show_ui'			=> false,
		);
		
		register_post_type( "ld-slack-{$notification_id}-feed", $args );
		
	}
	
	/**
	 * Handles Taxonomy Creation for Webhook Storage
	 * Taxonomies allow us to reference Term ID and to link the in-use Webhook directly to the Post
	 * 
	 * @param		string $notification_id   ID Used for Notification Hooks
	 * @param		array  $notification_args Array holding some basic Strings, but more importantly the Fields Array
	 *																										  
	 * @access		private
	 * @since		1.2.0
	 * @return		void
	 */
	private function create_webhook_taxonomies( $notification_id, $notification_args ) {
		
		$labels = array(
			'name' => "$notification_args[name] Webhooks",
			'singular_name'	=> "$notification_args[name] Webhook",
			'menu_name' => "$notification_args[name] Webhooks",
			'all_items' => "All $notification_args[name] Webhooks",
			'edit_item' => "Edit $notification_args[name] Webhook",
			'view_item' => "View $notification_args[name] Webhook",
			'update_item' => "Update $notification_args[name] Webhook",
			'add_new_item' => "Add New $notification_args[name] Webhook",
			'new_item_name' => "New $notification_args[name] Webhook Name",
			'parent_item' => "Parent $notification_args[name] Webhook",
			'parent_item_colon' => "Parent $notification_args[name] Webhook:",
			'search_items' => "Search $notification_args[name] Webhooks",
			'popular_items' => "Popular $notification_args[name] Webhooks",
			'separate_items_with_commas' => "Separate $notification_args[name] Webhooks with commas",
			'add_or_remove_items' => "Add or remove $notification_args[name] Webhooks",
			'choose_from_most_used' => "Choose from most used $notification_args[name] Webhooks",
			'name_admin_bar' => "$notification_args[name] Webhook",
			'not_found' => "No $notification_args[name] Webhooks found.",
		);
		
		$args = array(
			'labels' => $labels,
			'public' => false,
			'publicly_queryable' => true,
			'show_ui' => false,
		);
		
		register_taxonomy( "ld-slack-{$notification_id}-webhook", "ld-slack-{$notification_id}-feed", $args );
		
	}
	
	/**
	 * Route our Notification Triggers to something more specific
	 * 
	 * @param		string $trigger Trigger
	 * @param		array  $args	Arguments Array
	 *										
	 * @access		public
	 * @since		1.0.0
	 * @return		void
	 */
	public function notify( $trigger, $args ) {
		
		global $ld_slack_notifications;
		
		$ld_slack_notifications = apply_filters( 'ld_slack_notifications', array() );
		
		do_action( "ld_slack_notify_$trigger", $args );
		
		if ( $ld_slack_notifications ) {
			
			foreach ( $ld_slack_notifications as $notification_id => $notificaiton_args ) {
				
				$this->trigger_notification( $notification_id, $notificaiton_args, $trigger, $args );
				
			}
			
		}
		
	}
	
	/**
	 * Determines which specific route to send the Notitfication while grabbing related data
	 * 
	 * @param		string $notification_id		ID Used for Notification Hooks
	 * @param		array  $notification_args   $args defined by the ld_slack_notifications Filter
	 * @param		string $trigger				Notification Trigger
	 * @param		array  $args				$args Array passed from the original Trigger of the process
	 *																							  
	 * @access		private
	 * @since		1.0.0
	 * @return		void
	 */
	private function trigger_notification( $notification_id, $notification_args, $trigger, $args ) {
		
		$notifications = get_posts( array(
			'post_type'   => "ld-slack-{$notification_id}-feed",
			'numberposts' => -1,
			'meta_key'	=> "ld_slack_{$notification_id}_feed_trigger",
			'meta_value'  => $trigger,
		) );
		
		if ( ! $notifications ) {
			return;
		}
		
		foreach ( $notifications as $notification_post ) {
			
			// Get Field Keys
			$notification_post_values = array();
			foreach ( $notification_args['fields'] as $field_id => $field ) {
				
				if ( $field['type'] == 'hook' ) continue;
				
				$notification_post_values[ $field_id ] = get_post_meta(
					$notification_post->ID,
					"ld_slack_{$notification_id}_feed_$field_id",
					true
				);
				
			}
			
			// Fires the final Action that we use for whatever we feel like
			do_action( 
				"ld_slack_do_notification_$notification_id",
				$notification_post,
				$notification_post_values,
				$trigger,
				$notification_id,
				$args
			);
			
		}
		
	}
	
	/**
	 * Does String Replacements for common things like the name of the User who triggered the Notification
	 * 
	 * @param		array  $strings			Notification Fields to check for replacements in
	 * @param		array  $fields			Fields used to create the Post Meta
	 * @param		string $trigger			Notification Trigger
	 * @param		string $notification_id ID used for Notification Hooks
	 * @param		array  $args			$args Array passed from the original Trigger of the process
	 *	 
	 * @access		public
	 * @since		1.0.0
	 * @return		array  Replaced Strings within each Field
	 */
	public function notifications_replacements( $strings, $fields, $trigger, $notification_id, $args, $replacements = array() ) {
		
		$replacements = wp_parse_args( $replacements, array(
			'%username%' => '',
			'%name%' => '',
			'%email%' => '',
		) );
		
		if ( isset( $args['user_id'] ) ) {
		
			$userdata = get_userdata( $args['user_id'] );

			$replacements['%name%'] = $userdata->display_name;
			$replacements['%username%'] = $userdata->user_login;
			$replacements['%email%'] = $userdata->user_email;
			
		}
		
		/**
		* Allows additional replacements to be made.
		*
		* @since 1.0.0
		*/
		$replacements = apply_filters( 'ld_slack_notifications_replacements', $replacements, $fields, $trigger, $notification_id, $args );
		
		foreach ( $strings as $i => $string ) {
			$strings[ $i ] = str_replace( array_keys( $replacements ), array_values( $replacements ), $string );
		}
		
		return $strings;
		
	}

}