<?php
/*
Plugin Name: LearnDash LMS - iThemes Exchange
Plugin URI: http://www.learndash.com
Description: LearnDash iThemes Exchange Integration Plugin
Version: 1.1
Author: LearnDash
Author URI: http://www.learndash.com
License: GPL2
*/

/**
 * Copyright (c) 2014 LearnDash. All rights reserved.
 *
 * Released under the GPL license
 * http://www.opensource.org/licenses/gpl-license.php
 *
 * This is an add-on for WordPress
 * http://wordpress.org/
 *
 * **********************************************************************
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * **********************************************************************
 */

// don't call the file directly
if ( !defined( 'ABSPATH' ) ) exit;

/**
 * LearnDash_Exchange class
 *
 * @class LearnDash_Exchange The class that holds the entire LearnDash_Exchange plugin
 */
class LearnDash_Exchange {

    /**
     * Constructor for the LearnDash_Exchange class
     *
     * Sets up all the appropriate hooks and actions
     * within our plugin.
     *
     * @uses add_action()
     */
    public function __construct() {
        // Localize our plugin
        add_action( 'init', array( $this, 'localization_setup' ) );

        add_action( 'add_meta_boxes', array($this, 'add_meta_box') );

        // save the custom fields
        add_action( 'save_post', array( $this, 'save_exchange_course' ), 10, 2 );

        add_action( 'wp_head', array( $this, 'button_css' ) );

        // fires on exchange product status change
        add_action( 'it_exchange_update_transaction_status', array( $this, 'on_exchange_status_change' ) );
        add_action( 'it_exchange_add_transaction_success', array( $this, 'on_exchange_status_change' ) );

        // prints product link to course
        add_filter( 'learndash_payment_button', array( $this, 'show_course_product_link' ), 10, 2 );
    }

    /**
     * Initializes the LearnDash_Exchange() class
     *
     * Checks for an existing LearnDash_Exchange() instance
     * and if it doesn't find one, creates it.
     */
    public static function init() {
        static $instance = false;

        if ( ! $instance ) {
            $instance = new LearnDash_Exchange();
        }

        return $instance;
    }

    /**
     * Initialize plugin for localization
     *
     * @uses load_plugin_textdomain()
     */
    public function localization_setup() {
        load_plugin_textdomain( 'learndash-exchange', false, dirname( plugin_basename( __FILE__ ) ) . '/languages/' );
    }

    function add_meta_box() {
        add_meta_box( 'learndash-courses', __( 'LearnDash', 'wpuf' ), array($this, 'render_meta_box_courses'), 'it_exchange_prod', 'normal', 'high' );
        add_meta_box( 'learndash-exchange', __( 'iThemes Exchange', 'wpuf' ), array($this, 'render_meta_box_exchange'), 'sfwd-courses', 'side', 'low' );
    }

    /**
     * Prints meta box in exchange product area
     *
     * @return void
     */
    function render_meta_box_courses() {
        global $post;

        $courses = $this->get_post_list( 'sfwd-courses' );
        $selected = get_post_meta( $post->ID, '_learndash_courses', true );
        $selected = is_array( $selected ) ? $selected : array();
        ?>

        <input type="hidden" name="_learndash_courses_nonce" value="<?php echo wp_create_nonce( plugin_basename( __FILE__ ) ); ?>" />

        <label for="_learndash_courses">
            <?php _e( 'Courses', 'learndash-exchange' ); ?>
            <span class="tip" title="<?php esc_attr_e( 'You can select multiple courses to sell together holding the SHIFT key when clicking.', 'learndash-exchange' ); ?>">i</span>
        </label>

        <select name="_learndash_courses[]" id="_learndash_courses" multiple="multiple">
            <?php foreach ($courses as $course_id => $course_name) { ?>
                <option value="<?php echo esc_attr( $course_id ); ?>"<?php selected( in_array( $course_id, $selected), true ); ?>><?php echo esc_html( $course_name ); ?></option>
            <?php } ?>
        </select>

        <?php
    }

    /**
     * Prints meta box in exchange product area
     *
     * @return void
     */
    function render_meta_box_exchange() {
        global $post;

        $products = $this->get_post_list( 'it_exchange_prod' );
        $selected = get_post_meta( $post->ID, '_exchange_product', true );
        $selected = is_array( $selected ) ? $selected : array();
        ?>

        <input type="hidden" name="_learndash_courses_nonce" value="<?php echo wp_create_nonce( plugin_basename( __FILE__ ) ); ?>" />

        <label for="_exchange_product">
            <?php _e( 'Exchange Product', 'learndash-exchange' ); ?>
        </label>

        <select name="_exchange_product[]" id="_exchange_product" multiple="multiple">
            <?php foreach ($products as $product_id => $product_name) { ?>
                <option value="<?php echo esc_attr( $product_id ); ?>"<?php selected( in_array( $product_id, $selected), true ); ?>><?php echo esc_html( $product_name ); ?></option>
            <?php } ?>
        </select>

        <p class="description"><?php _e( 'Linked iThemes Exchange product with this course', 'learndash-exchange' ); ?></p>

        <?php
    }

    /**
     * Retrieve or display list of posts as a dropdown (select list).
     *
     * @return string HTML content, if not displaying.
     */
    function get_post_list( $post_type ) {

        $array = array();
        $courses = get_posts( array('post_type' => $post_type, 'posts_per_page' => -1) );
        if ( $courses ) {
            foreach ($courses as $course) {
                $array[$course->ID] = $course->post_title;
            }
        }

        return $array;
    }


    /**
     * Saves the course ID from selection meta box
     *
     * @param int $post_id
     * @param object $post
     * @return int|void
     */
    function save_exchange_course( $post_id, $post ) {

        if ( isset( $_POST['_learndash_courses_nonce'] ) && !wp_verify_nonce( $_POST['_learndash_courses_nonce'], plugin_basename( __FILE__ ) ) ) {
            return;
        }

        // Is the user allowed to edit the post or page?
        if ( !current_user_can( 'edit_post', $post->ID ) ) {
            return $post->ID;
        }

        if ( $post->post_type == 'it_exchange_prod' ) {
            // update the courses
            $courses = isset( $_POST['_learndash_courses'] ) ? $_POST['_learndash_courses'] : array();
            update_post_meta( $post->ID, '_learndash_courses', $courses );

            // update each courses
            $this->update_course_association( $post->ID, $courses );


        } else if ( $post->post_type == 'sfwd-courses' ) {
            $product_ids = isset( $_POST['_exchange_product'] ) ? $_POST['_exchange_product'] : array();

            update_post_meta( $post->ID, '_exchange_product', $product_ids );
            $this->update_product_association( $post->ID, $product_ids );
        }


    }

    /**
     * Update each courses with exchange product ID from Product UI
     *
     * We are updating each courses to make aware that this course has
     * a exchange product associated. This is neccessary for showing payment
     * button or link to product in course post type in frontend.
     *
     * @param int $product_id
     * @param array $courses
     */
    function update_course_association( $product_id, $courses = array() ) {
        if ( $courses ) {
            foreach ($courses as $course_id) {
                $existing = get_post_meta( $course_id, '_exchange_product', true );
                $existing = is_array( $existing ) ? $existing : array();

                // only add if it's not exists
                if ( !in_array( $product_id, $existing) ) {
                    $existing[] = $product_id;
                }

                update_post_meta( $course_id, '_exchange_product', $existing );
            }
        }
    }

    /**
     * Update the product with associated course from Course UI
     *
     * @param int $course_id
     * @param int $product_id
     */
    function update_product_association( $course_id, $product_id ) {
        $courses = get_post_meta( $product_id, '_learndash_courses', true );
        $courses = is_array( $courses ) ? $courses : array();

        // only add if it's not exists
        if ( !in_array( $course_id, $courses) ) {
            $courses[] = $course_id;
        }

        update_post_meta( $product_id, '_learndash_courses', $courses );
    }

    /**
     * Show the product link button in course frontend
     *
     * @global object $post
     * @param string $payment_link
     * @return string
     */
    function show_course_product_link( $payment_link ) {
        global $post;

        $product_ids = get_post_meta( $post->ID, '_exchange_product', true );
        $product_ids = is_array( $product_ids ) ? $product_ids : array();

        $products = count( $product_ids );
        if ( $products  ) {
            if ( $products == 1 ) {
                $url = get_permalink( $product_ids[0] );
            } else {
                $url = it_exchange_get_page_url( 'product' );
            }

            $payment_link = sprintf( '<a class="ld-ithemes-exchange" href="%s">%s</a>', $url, __( 'Take this Course', 'learndash-exchange' ) );
        }

        return $payment_link;
    }

    /**
     * Handles iThemes Exchange
     *
     * @param object $transaction
     */
    function on_exchange_status_change( $transaction_id ) {

        $transaction = it_exchange_get_transaction( $transaction_id );
        $products = it_exchange_get_transaction_products( $transaction );
        $customer_id = it_exchange_get_transaction_customer_id( $transaction->ID );
        $status = $transaction->get_status();

        switch ( $status ) {
            case 'Completed':
            case 'paid':
            case 'succeeded':
                foreach ($products as $product) {
                    $this->grant_user_access( $product['product_id'], $customer_id, false );
                }

                break;

            case 'refunded':
                // remove enrollment if status is not paid
                foreach ($products as $product) {
                    $this->grant_user_access( $product['product_id'], $customer_id, true );
                }

                break;
        }
    }

    /**
     * Grant user access based on product id
     *
     * Fetches all courses that a product has. Then updates each course access
     * to the user.
     *
     * @param int $product_id
     * @param int $customer_id
     * @param bool $access
     */
    function grant_user_access( $product_id, $customer_id, $access ) {
        $courses = get_post_meta( $product_id, '_learndash_courses', true );
        $courses = is_array( $courses ) ? $courses : array();

        if ( $courses ) {
            foreach ($courses as $course_id) {
                ld_update_course_access( $customer_id, $course_id, $access );
            }
        }
    }

    function button_css() {
        ?>
        <style type="text/css">
            a.ld-ithemes-exchange,
            a.ld-ithemes-exchange:active,
            a.ld-ithemes-exchange:visited,
            a.ld-ithemes-exchange:link {
                padding: 8px 15px;
                font-size: 15px;
                line-height: 1.471;
                border-radius: 6px;
                display: inline-block;
                color: #ffffff;
                background-color: #2ecc71;
                border: none;
                text-decoration: none;
            }
        </style>
        <?php
    }


} // LearnDash_Exchange

LearnDash_Exchange::init();
