<?php

namespace uncanny_learndash_groups;

// If this file is called directly, abort.
if ( ! defined( 'WPINC' ) ) {
	die;
}

/**
 * This class is used to run any configurations before the plugin is initialized
 *
 * @package    uncanny_learndash_groups
 * @subpackage uncanny_learndash_groups/config
 * @author     Uncanny Owl
 */
class Config {

	/**
	 * @var string
	 */
	public static $invalid_code;
	/**
	 * @var string
	 */
	public static $already_redeemed;
	/**
	 * @var string
	 */
	public static $redeemed_maximum;
	/**
	 * @var string
	 */
	public static $successfully_redeemed;
	/**
	 * The instance of the class
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      Boot
	 */
	private static $instance = null;

	/**
	 * Creates singleton instance of class
	 *
	 * @return Config $instance The Config Class
	 * @since 1.0.0
	 *
	 */
	public static function get_instance() {

		if ( self::$instance === null ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Config constructor.
	 */
	function __construct() {
		self::$invalid_code          = esc_html__( 'Sorry, the code you entered is not valid.', 'uncanny-learndash-groups' );
		self::$already_redeemed      = esc_html__( 'Sorry, the code you entered has already been redeemed.', 'uncanny-learndash-groups' );
		self::$redeemed_maximum      = esc_html__( 'Sorry, the code you entered has already been redeemed maximum times.', 'uncanny-learndash-groups' );
		self::$successfully_redeemed = esc_html__( 'Congratulations, the code you entered has successfully been redeemed.', 'uncanny-learndash-groups' );
	}

	/**
	 * Initialize the class and setup its properties.
	 *
	 * @param string $plugin_name The name of the plugin
	 * @param string $prefix The variable used to prefix filters and actions
	 * @param string $version The version of this plugin
	 * @param string $file The main plugin file __FILE__
	 * @param bool $debug Whether debug log in php and js files are enabled
	 *
	 * @since    1.0.0
	 *
	 */
	public function configure_plugin_before_boot( $plugin_name, $prefix, $version, $file, $debug ) {


		$this->define_constants( $plugin_name, $prefix, $version, $file, $debug );

		do_action( Utilities::get_prefix() . '_define_constants_after' );

		register_activation_hook( Utilities::get_plugin_file(), array( $this, 'activation' ) );

		register_deactivation_hook( Utilities::get_plugin_file(), array( $this, 'deactivation' ) );

		do_action( Utilities::get_prefix() . '_config_setup_after' );

	}

	/**
	 *
	 * This action is documented in includes/class-plugin-name-deactivator.php
	 *
	 * @param string $plugin_name The name of the plugin
	 * @param string $prefix Variable used to prefix filters and actions
	 * @param string $version The version of this plugin.
	 * @param string $plugin_file The main plugin file __FILE__
	 * @param string $debug_mode Whether debug log in php and js files are enabled
	 *
	 * @since    1.0.0
	 * @access   private
	 *
	 */
	private function define_constants( $plugin_name, $prefix, $version, $plugin_file, $debug_mode ) {


		// Set and define version
		if ( ! defined( strtoupper( $prefix ) . '_PLUGIN_NAME' ) ) {
			define( strtoupper( $prefix ) . '_PLUGIN_NAME', $plugin_name );
			Utilities::set_plugin_name( $plugin_name );
		}

		// Set and define version
		if ( ! defined( strtoupper( $prefix ) . '_VERSION' ) ) {
			define( strtoupper( $prefix ) . '_VERSION', $version );
			Utilities::set_version( $version );
		}

		// Set and define prefix
		if ( ! defined( strtoupper( $prefix ) . '_PREFIX' ) ) {
			define( strtoupper( $prefix ) . '_PREFIX', $prefix );
			Utilities::set_prefix( $prefix );
		}

		// Set and define the main plugin file path
		if ( ! defined( $prefix . '_FILE' ) ) {
			define( strtoupper( $prefix ) . '_FILE', $plugin_file );
			Utilities::set_plugin_file( $plugin_file );
		}

		// Set and define debug mode
		if ( ! defined( $prefix . '_DEBUG_MODE' ) ) {
			define( strtoupper( $prefix ) . '_DEBUG_MODE', $debug_mode );
			Utilities::set_debug_mode( $debug_mode );
		}

		// Set and define the server initialization time
		if ( ! defined( $prefix . '_SERVER_INITIALIZATION' ) ) {
			$time = time();
			define( strtoupper( $prefix ) . '_SERVER_INITIALIZATION', $time );
			Utilities::set_plugin_initialization( $time );
		}

		Utilities::log(
			array(
				'get_plugin_name'           => Utilities::get_plugin_name(),
				'get_version'               => Utilities::get_version(),
				'get_prefix'                => Utilities::get_prefix(),
				'get_plugin_file'           => Utilities::get_plugin_file(),
				'get_debug_mode'            => Utilities::get_debug_mode(),
				'get_plugin_initialization' => date( Utilities::get_date_time_format(), Utilities::get_plugin_initialization() )

			),
			'Configuration Variables'
		);

	}


	/**
	 * The code that runs during plugin activation.
	 * @since    1.0.0
	 */
	function activation() {

		do_action( Utilities::get_prefix() . '_activation_before' );

		// Set which roles will need access
		$set_role_capabilities = array(
			'group_leader'  => array( 'ulgm_group_management' ),
			'administrator' => array( 'ulgm_group_management' ),
		);

		/**
		 * Filters role based capabilities before being added
		 *
		 * @param string $set_role_capabilities Path to the plugins template folder
		 *
		 * @since 1.0.0
		 *
		 */
		$set_role_capabilities = apply_filters( Utilities::get_prefix() . '_add_role_capabilities', $set_role_capabilities );

		include_once( Utilities::get_include( 'capabilities.php' ) );
		$capabilities = new Capabilities( $set_role_capabilities );
		$capabilities->add_capabilities();

		include_once( Utilities::get_include( 'database.php' ) );
		Database::create_tables();
		Database::alter_db_table();

		do_action( Utilities::get_prefix() . '_activation_after' );

		$user_id = get_current_user_id();

		$create_group_page = array(
			'post_type'    => 'page',
			'post_title'   => 'Group Management',
			'post_content' => '[uo_groups]',
			'post_status'  => 'publish',
			'post_author'  => $user_id,
			'post_name'    => 'group-management',
		);
		$page_id_exists    = $this->the_slug_exists( 'group-management' );
		if ( false === $page_id_exists ) {
			$management_page_id = wp_insert_post( $create_group_page );
		} else {
			$management_page_id = $page_id_exists;
		}
		update_option( 'ulgm_group_management_page', $management_page_id );

		//
		$create_report_page = array(
			'post_type'    => 'page',
			'post_title'   => 'Group Management Report',
			'post_content' => '[uo_groups_course_report]',
			'post_status'  => 'publish',
			'post_author'  => $user_id,
			'post_name'    => 'group-management-report',
		);

		$page_id_exists = $this->the_slug_exists( 'group-management-report' );
		if ( false === $page_id_exists ) {
			$report_page_id = wp_insert_post( $create_report_page );
		} else {
			$report_page_id = $page_id_exists;
		}
		update_option( 'ulgm_group_report_page', $report_page_id );

		//
		$create_quiz_report_page = array(
			'post_type'    => 'page',
			'post_title'   => 'Group Quiz Report',
			'post_content' => '[uo_groups_quiz_report]',
			'post_status'  => 'publish',
			'post_author'  => $user_id,
			'post_name'    => 'group-quiz-report',
		);

		$page_id_exists = $this->the_slug_exists( 'group-quiz-report' );
		if ( false === $page_id_exists ) {
			$quiz_report_page_id = wp_insert_post( $create_quiz_report_page );
		} else {
			$quiz_report_page_id = $page_id_exists;
		}
		update_option( 'ulgm_group_quiz_report_page', $quiz_report_page_id );

		//
		$create_assignment_report_page = [
			'post_type'    => 'page',
			'post_title'   => 'Assignment Management Page',
			'post_content' => '[uo_groups_assignments]',
			'post_status'  => 'publish',
			'post_author'  => $user_id,
			'post_name'    => 'assignment-management-page',
		];

		$page_id_exists = $this->the_slug_exists( 'assignment-management-page' );
		if ( false === $page_id_exists ) {
			$assignment_report_page_id = wp_insert_post( $create_assignment_report_page );
		} else {
			$assignment_report_page_id = $page_id_exists;
		}
		update_option( 'ulgm_group_assignment_report_page', $assignment_report_page_id );

		//
		$create_essay_report_page = array(
			'post_type'    => 'page',
			'post_title'   => 'Essay Management Page',
			'post_content' => '[uo_groups_essays]',
			'post_status'  => 'publish',
			'post_author'  => $user_id,
			'post_name'    => 'essay-management-page',
		);

		$page_id_exists = $this->the_slug_exists( 'essay-management-page' );
		if ( false === $page_id_exists ) {
			$essay_report_page_id = wp_insert_post( $create_essay_report_page );
		} else {
			$essay_report_page_id = $page_id_exists;
		}
		update_option( 'ulgm_group_essay_report_page', $essay_report_page_id );

		//
		if ( Utilities::if_woocommerce_active() ) {
			$create_a_la_carte_license_page = array(
				'post_type'    => 'page',
				'post_title'   => 'Group Management Buy Courses',
				'post_content' => '[uo_groups_buy_courses]',
				'post_status'  => 'publish',
				'post_author'  => $user_id,
				'post_name'    => 'group-management-buy-courses',
			);

			$page_id_exists = $this->the_slug_exists( 'group-management-buy-courses' );
			if ( false === $page_id_exists ) {
				$buy_courses_id = wp_insert_post( $create_a_la_carte_license_page );
			} else {
				$buy_courses_id = $page_id_exists;
			}
			update_option( 'ulgm_group_buy_courses_page', $buy_courses_id );
		}
	}

	/**
	 * The code that runs during plugin deactivation.
	 * @since    1.0.0
	 */
	function deactivation() {

		do_action( Utilities::get_prefix() . '_deactivation_before' );

		// Set which roles will need access
		$set_role_capabilities = array(
			'group_leader'  => array( 'ulgm_group_management' ),
			'administrator' => array( 'ulgm_group_management' ),
		);

		/**
		 * Filters role based capabilities before being added
		 *
		 * @param string $set_role_capabilities Path to the plugins template folder
		 *
		 * @since 1.0.0
		 *
		 */
		$set_role_capabilities = apply_filters( Utilities::get_prefix() . '_add_role_capabilities', $set_role_capabilities );

		include_once( Utilities::get_include( 'capabilities.php' ) );
		$capabilities = new Capabilities( $set_role_capabilities );
		$capabilities->remove_capabilities();

		do_action( Utilities::get_prefix() . '_deactivation_after' );

		$slug    = 'group-management';
		$post_id = $this->the_slug_exists( $slug );
		//Utilities::log( $post_id, 'POst ID' );
		if ( is_integer( $post_id ) ) {
			wp_update_post( array( 'ID' => $post_id, 'post_status' => 'draft', ) );
		}
	}

	/**
	 * @param $post_name
	 *
	 * @return bool|int
	 */
	function the_slug_exists( $post_name ) {
		global $wpdb;
		$qry = $wpdb->get_var( "SELECT ID FROM {$wpdb->posts} WHERE post_name LIKE '{$post_name}'" );
		Utilities::log(["SELECT ID FROM {$wpdb->posts} WHERE post_name LIKE '{$post_name}'", $post_name, $qry], 'qry, $post_name, $qry', true, 'page');
		if ( $qry ) {
			return (int) $qry;
		} else {
			return false;
		}
	}
}
