<?php

namespace uncanny_learndash_groups;

// If this file is called directly, abort.
use calderawp\calderaforms\cf2\Exception;

if ( ! defined( 'WPINC' ) ) {
	die;
}

/**
 * Class Boot
 * @package uncanny_learndash_groups
 */
class Boot {

	/**
	 * The instance of the class
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      Boot
	 */
	private static $instance = null;

	/**
	 * The directories that are auto loaded and initialized
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      array
	 */
	private static $auto_loaded_directories = null;

	/**
	 * class constructor
	 */
	private function __construct() {

		// We need to check if spl auto loading is available when activating plugin
		// Plugin will not activate if SPL extension is not enabled by throwing error
		if ( ! extension_loaded( 'SPL' ) ) {
			trigger_error( esc_html__( 'Please contact your hosting company to update to php version 5.3+ and enable spl extensions.', 'uncanny-learndash-groups' ), E_USER_ERROR );
		}

		spl_autoload_register( array( $this, 'require_class_files' ) );

		// Initialize all classes in given directories
		$this->auto_initialize_classes();
		add_action( 'admin_init', array( $this, 'woocommerce_modify_license_product_type' ) );

		// Import Gutenberg Blocks
		require_once( dirname( __FILE__ ) . '/blocks/blocks.php' );
		new Blocks();

		add_action( 'admin_menu', array( __CLASS__, 'add_help_submenu' ), 51 );
		add_action( 'admin_menu', array( __CLASS__, 'add_uncanny_plugins_page' ), 52 );

		add_action( 'admin_enqueue_scripts', array( __CLASS__, 'enqueue_external_scripts' ) );

		add_action( 'admin_init', array( __CLASS__, 'uo_admin_help_process' ) );
	}

	/**
	 *
	 */
	public function woocommerce_modify_license_product_type() {
		if ( 'no' === get_option( 'ulgm_license_product_updated', 'no' ) ) {
			$users = new \WP_User_Query( array(
				'role__in' => array(
					'shop_manager',
					'administrator',
				),
			) );

			$results = $users->get_results();

			$user_ids = array();
			foreach ( $results as $result ) {
				$user_ids[] = (int) $result->ID;
			}
			$user_ids = ! empty( $user_ids ) ? $user_ids : PHP_INT_MAX;

			$posts = get_posts( array(
				'post_type'      => 'product',
				'posts_per_page' => 999,
				'author__not_in' => $user_ids,
				'tax_query'      =>
					array(
						array(
							'taxonomy' => 'product_type',
							'field'    => 'slug',
							'terms'    => 'license',
							'operator' => 'IN',
						),
					),
			) );
			
			if ( $posts ) {
				foreach ( $posts as $post ) {
					wp_set_object_terms( $post->ID, array(
						'exclude-from-catalog',
						'exclude-from-search',
					), 'product_visibility' );
				}
			}
			update_option( 'ulgm_license_product_updated', 'yes' );
		}

	}


	/**
	 * Creates singleton instance of Boot class and defines which directories are auto loaded
	 *
	 * @since 1.0.0
	 *
	 * @param array $auto_loaded_directories
	 *
	 * @return Boot
	 */
	public static function get_instance( $auto_loaded_directories = array( 'classes/' ) ) {

		if ( null === self::$instance ) {

			// Define directories were the auto loader looks for files and initializes them
			self::$auto_loaded_directories = $auto_loaded_directories;

			// Lets boot up!
			self::$instance = new self();
		}

		return self::$instance;
	}


	/**
	 * SPL Auto Loader functions
	 *
	 * @since 1.0.0
	 *
	 * @param string $class Any
	 */
	public function require_class_files( $class ) {

		// Remove Class's namespace eg: my_namespace/MyClassName to MyClassName
		$class = str_replace( __NAMESPACE__, '', $class );
		$class = str_replace( '\\', '', $class );

		// First Character of class name to lowercase eg: MyClassName to myClassName
		$class_to_filename = lcfirst( $class );

		// Split class name on upper case letter eg: myClassName to array( 'my', 'Class', 'Name')
		$split_class_to_filename = preg_split( '#([A-Z][^A-Z]*)#', $class_to_filename, null, PREG_SPLIT_DELIM_CAPTURE | PREG_SPLIT_NO_EMPTY );

		if ( 1 <= count( $split_class_to_filename ) ) {
			// Split class name to hyphenated name eg: array( 'my', 'Class', 'Name') to my-Class-Name
			$class_to_filename = implode( '-', $split_class_to_filename );
		}

		// Create file name that will be loaded from the classes directory eg: my-Class-Name to my-class-name.php
		$file_name = strtolower( $class_to_filename ) . '.php';


		// Check each directory
		foreach ( self::$auto_loaded_directories as $directory ) {

			// add a filter to override file
			$file_path = dirname( __FILE__ ) . DIRECTORY_SEPARATOR . $directory . $file_name;

			// Does the file exist
			if ( file_exists( $file_path ) ) {

				// File found, require it
				require_once( $file_path );

				// You can cannot have duplicate files names. Once the first file is found, the loop ends.
				return;
			}
		}

	}

	/**
	 * Looks through all defined directories and modifies file name to create new class instance.
	 *
	 * @since 1.0.0
	 *
	 */
	private function auto_initialize_classes() {

		// Check each directory
		foreach ( self::$auto_loaded_directories as $directory ) {

			// Get all files in directory
			$files = scandir( dirname( __FILE__ ) . DIRECTORY_SEPARATOR . $directory );

			// remove parent directory, sub directory, and silence is golden index.php
			$files = array_diff( $files, array( '..', '.', 'index.php' ) );

			// Loop through all files in directory to create class names from file name
			foreach ( $files as $file ) {

				// Remove file extension my-class-name.php to my-class-name
				$file_name = str_replace( '.php', '', $file );

				// Split file name on - eg: my-class-name to array( 'my', 'class', 'name')
				$class_to_filename = explode( '-', $file_name );

				// Make the first letter of each word in array upper case - eg array( 'my', 'class', 'name') to array( 'My', 'Class', 'Name')
				$class_to_filename = array_map( function ( $word ) {
					return ucfirst( $word );
				}, $class_to_filename );

				// Implode array into class name - eg. array( 'My', 'Class', 'Name') to MyClassName
				$class_name = implode( $class_to_filename );

				$class = __NAMESPACE__ . '\\' . $class_name;

				$obj = new $class;
				Utilities::set_class_instance( $class_name, $obj );

			}
		}
	}

	/**
	 * Add "Help" submenu
	 */
	public static function add_help_submenu() {
		add_submenu_page(
			'uncanny-groups-create-group',
			__( 'Uncanny LearnDash Groups Support', 'uncanny-learndash-groups' ),
			__( 'Help', 'uncanny-learndash-groups' ),
			'manage_options',
			'uncanny-groups-kb',
			array( __CLASS__, 'include_help_page' )
		);
	}

	/**
	 * Create "Uncanny Plugins" submenu
	 */
	public static function add_uncanny_plugins_page() {
		add_submenu_page(
			'uncanny-groups-create-group',
			__( 'Uncanny LearnDash Plugins', 'uncanny-learndash-groups' ),
			__( 'LearnDash Plugins', 'uncanny-learndash-groups' ),
			'manage_options',
			'uncanny-groups-plugins',
			array( __CLASS__, 'include_learndash_plugins_page' )
		);
	}

	/**
	 * Include "Help" template
	 */
	public static function include_help_page() {
		include( 'templates/admin-help.php' );
	}

	/**
	 * Include "LearnDash Plugins" template
	 */
	public static function include_learndash_plugins_page() {
		include( 'templates/admin-learndash-plugins.php' );
	}

	/**
	 * Enqueue external scripts from uncannyowl.com
	 */
	public static function enqueue_external_scripts() {
		$pages_to_include = [ 'uncanny-groups-plugins', 'uncanny-groups-kb' ];

		if ( isset( $_GET[ 'page' ] ) && in_array( $_GET[ 'page' ], $pages_to_include ) ){
			wp_enqueue_style( 'uncannyowl-core', 'https://uncannyowl.com/wp-content/mu-plugins/uncanny-plugins-core/dist/bundle.min.css', array(), UNCANNY_GROUPS_VERSION );

			wp_enqueue_script( 'uncannyowl-core', 'https://uncannyowl.com/wp-content/mu-plugins/uncanny-plugins-core/dist/bundle.min.js', array( 'jquery' ), UNCANNY_GROUPS_VERSION );
		}
	}

	/**
	 * Submit ticket
	 */
	public static function uo_admin_help_process() {
		if ( isset( $_POST[ 'ulgm-send-ticket' ] ) && check_admin_referer( 'uncanny0w1', 'ulgm-send-ticket' ) ) {
			$name     = esc_html( $_POST['fullname'] );
			$email    = esc_html( $_POST['email'] );
			$website  = esc_html( $_POST['website'] );
			$message  = esc_html( $_POST['message'] );
			$siteinfo = stripslashes( $_POST['siteinfo'] );
			if ( isset( $_POST['site-data'] ) && 'yes' === $_POST['site-data'] ) {
				$message = "<h3>Message:</h3><p>{$message}</p><br /><hr /><h3>User Site Information:</h3>{$siteinfo}";
			}

			$to        = 'support.41077.bb1dda3d33afb598@helpscout.net';
			$subject   = esc_html( $_POST['subject'] );
			$headers   = array( 'Content-Type: text/html; charset=UTF-8' );
			$headers[] = 'From: ' . $name . ' <' . $email . '>';
			$headers[] = 'Reply-To:' . $name . ' <' . $email . '>';
			wp_mail( $to, $subject, $message, $headers );
			if ( isset( $_POST['page'] ) ) {
				$url = admin_url( 'admin.php' ) . '?page=' . esc_html( $_POST['page'] ) . '&sent=true&wpnonce=' . wp_create_nonce();
				wp_safe_redirect( $url );
				exit;
			}
		}
	}

	/**
	 * Make clone magic method private, so nobody can clone instance.
	 *
	 * @since 1.0.0
	 */
	private function __clone() {
	}

	/**
	 * Make sleep magic method private, so nobody can serialize instance.
	 *
	 * @since 1.0.0
	 */
	public function __sleep() {
	}

	/**
	 * Make wakeup magic method private, so nobody can unserialize instance.
	 *
	 * @since 1.0.0
	 */
	private function __wakeup() {

	}
}