<?php
/**
 * ConvertKit API class
 *
 * @since 1.0.0
 */

/**
 * ConvertKit_API Class
 *
 * Establishes API connection to ConvertKit App
 *
 * @since 1.0.0
 */
class LD_ConvertKit_API {

	/**
	 * ConvertKit API Key
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	protected $api_key;

	/**
	 * ConvertKit API Secret
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	protected $api_secret;

	/**
	 * Version of ConvertKit API
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	protected $api_version = 'v3';

	/**
	 * ConvertKit API URL
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	protected $api_url_base = 'https://api.convertkit.com/';


	/**
	 * Constructor for ConvertKitAPI instance
	 *
	 * @param string $api_key ConvertKit API Key.
	 * @param string $api_secret ConvertKit API Secret.
	 */
	public function __construct( $api_key, $api_secret ) {

		$this->api_key    = $api_key;
		$this->api_secret = $api_secret;
	}

	/**
	 * Gets account info.
	 *
	 * @since 1.0.0
	 *
	 * @return array|int 0 if error, array if success.
	 */
	public function get_account_info() {

		$result = $this->get_api_response( 'account' );

		if ( is_wp_error( $result ) ) {

			return 0;
		}

		if ( ! isset( $result['error'] ) ) {

			return $result;
		}

		return 0;
	}

	/**
	 * Gets all forms.
	 *
	 * @since 1.0.0
	 *
	 * @return array|int List of forms or 0 on failure.
	 */
	public function get_forms() {

		$result = $this->get_api_response( 'forms' );

		if ( isset( $result['error'] ) || ! isset( $result['forms'] ) ) {

			return 0;
		}

		return $result['forms'];
	}

	/**
	 * Get a form's subscriptions.
	 *
	 * @since 1.0.0
	 *
	 * @param int|string $form_id ID of form to lookup.
	 *
	 * @return array|int 0 on failure or an array of subscribers.
	 */
	public function get_form_subscriptions( $form_id ) {

		$subscribers = array();
		$page        = 0;
		$total_pages = 1;

		while ( $page < $total_pages ) {

			$result = $this->get_api_response( "forms/{$form_id}/subscriptions" );

			if ( isset( $result['error'] ) ) {

				return 0;
			}

			$subscribers = $subscribers + $result['subscriptions'];

			$page        = (int) $result['page'];
			$total_pages = (int) $result['total_pages'];
		}

		return $subscribers;
	}

	/**
	 * Get a subscriber by email address.
	 *
	 * @since 1.0.0
	 *
	 * @param string $email
	 *
	 * @return array|int|bool Subscriber data array on success, false if cannot find subscriber, 0 on failure.
	 */
	public function get_subscriber_by_email( $email ) {

		$result = $this->get_api_response( 'subscribers', array(
			'email_address' => $email,
		) );

		if ( isset( $result['error'] ) ) {

			return 0;
		}

		if ( $result['total_subscribers'] === 0 ) {

			return false;
		}

		return $result['subscribers'][0];
	}

	/**
	 * Get a subscriber's tags.
	 *
	 * @since 1.0.0
	 *
	 * @param string|int $subscriber_id
	 *
	 * @return array|int|bool Subscriber tags array on success, 0 on failure.
	 */
	public function get_subscriber_tags( $subscriber_id ) {

		$result = $this->get_api_response( "subscribers/{$subscriber_id}/tags" );

		if ( isset( $result['error'] ) || ! isset( $result['tags'] ) ) {

			return 0;
		}

		return $result['tags'];
	}

	/**
	 * Adds a subscriber to a Form.
	 *
	 * @since 1.0.0
	 *
	 * @param string $form_id Form ID.
	 * @param array $options Array of user data.
	 *
	 * @return int|bool Subscriber ID if success, false if failure
	 */
	public function form_subscribe( $form_id, $options ) {

		$request = "forms/{$form_id}/subscribe";

		$args = array(
			'email'      => $options['email'],
			'first_name' => $options['first_name'],
		);

		if ( isset( $options['tags'] ) ) {

			$args['tags'] = is_array( $options['tags'] ) ? implode( ',', $options['tags'] ) : (string) $options['tags'];
		}

		if ( isset( $options['fields'] ) ) {

			$args['fields'] = $options['fields'];
		}

		$result = $this->make_request( $request, 'POST', $args );

		return $result['status'] === 'success' ? (int) $result['response']['subscription']['subscriber']['id'] : false;
	}

	/**
	 * Adds a subscriber to a Tag.
	 *
	 * @since 1.0.0
	 *
	 * @param string $tag_id Tag ID.
	 * @param array $options Array of user data.
	 *
	 * @return bool True on success, false on failure.
	 */
	public function tag_subscribe( $tag_id, $options ) {

		$request = "tags/{$tag_id}/subscribe";

		$args = array(
			'email'      => $options['email'],
			'first_name' => $options['first_name'],
		);

		if ( isset( $options['fields'] ) ) {

			$args['fields'] = $options['fields'];
		}

		$result = $this->make_request( $request, 'POST', $args );

		return $result['status'] === 'success';
	}

	/**
	 * Get the ConvertKit subscriber ID associated with email address if it exists.
	 * Return 0 if subscriber not found.
	 *
	 * @param $email_address
	 *
	 * @return int $subscriber_id
	 */
	public function get_subscriber_id( $email_address ) {

		$url = add_query_arg(
			array(
				'api_secret'    => WP_ConvertKit::get_api_secret(),
				'email_address' => $email_address,
				'status'        => 'all',
			),
			'https://api.convertkit.com/v3/subscribers'
		);

		$result = $this->get_api_response( 'subscribers', array(
			'email_address' => $email_address,
			'status'        => 'all',
		) );

		if ( is_wp_error( $result ) ) {

			return 0;
		}

		$subs = json_decode( $result );

		$subscribers = is_array( $subs->subscribers ) ? $subs->subscribers : array();

		if ( $subscribers ) {

			$subscriber = array_pop( $subscribers );

			return $subscriber->id;
		}

		// subscriber not found
		return 0;

	}

	/**
	 * Gets subscriber info.
	 *
	 * @since 1.1.0
	 *
	 * @param string|int $subscriber_id
	 *
	 * @return array|int Subscriber array on success, 0 on failure.
	 */
	public function get_subscriber( $subscriber_id ) {

		$result = $this->get_api_response( "subscribers/{$subscriber_id}" );

		if ( is_wp_error( $result ) ) {

			return 0;
		}

		if ( isset( $result['subscriber'] ) ) {

			return $result['subscriber'];
		}

		// subscriber not found
		return 0;
	}

	/**
	 * Modifies a subscriber.
	 *
	 * @since 1.1.0
	 *
	 * @param string|int $subscriber_id
	 * @param array $args Fields to update.
	 *
	 * @return array Response from API.
	 */
	public function update_subscriber( $subscriber_id, $args ) {

		$request = "subscribers/{$subscriber_id}";

		$args = array_filter( wp_parse_args( $args, array(
			'email'      => null,
			'first_name' => null,
			'fields'     => null,
		) ) );

		$result = $this->make_request( $request, 'PUT', $args );

		return $result;
	}

	/**
	 * Creates a tag.
	 *
	 * @since 1.0.0
	 *
	 * @param array $tag_data
	 *
	 * @return array|int $tag_data Array of tag data on success, 0 on failure, tag ID if already exists.
	 */
	public function create_tag( $tag_data ) {

		$result = $this->make_request( 'tags', 'POST', array(
			'tag' => $tag_data,
		) );

		if ( $result['status'] === 'failed' ) {

			return 0;
		}

		return $result['response'];
	}

	/**
	 * Edits a tag.
	 *
	 * @since 1.0.0
	 *
	 * @param string|int $tag_id
	 * @param array $tag_data
	 *
	 * @return array|int Array of tag data on success, 0 on failure.
	 */
	public function edit_tag( $tag_id, $tag_data ) {

		// TODO If ConvertKit ever enables this feature, get it working.
		return 0;
	}

	/**
	 * Gets a tag by ID.
	 *
	 * @since 1.0.0
	 *
	 * @param string|int $tag_id Tag to lookup.
	 *
	 * @return array|int Array of tag data on success, 0 on failure.
	 */
	public function get_tag( $tag_id ) {

		$result = $this->get_api_response( "tags/{$tag_id}" );

		if ( isset( $result['error'] ) ) {

			return 0;
		}

		return $result;
	}

	/**
	 * Gets all tags for the account.
	 *
	 * @since 1.0.0
	 */
	public function get_all_tags() {

		$result = $this->get_api_response( 'tags' );

		if ( $result['status'] === 'failed' ) {

			return 0;
		}

		return $result['tags'];
	}

	/**
	 * Gets a custom field.
	 *
	 * @since 1.1.0
	 *
	 * @param string $field_label Field label.
	 *
	 * @return int|false Field ID if success, false if fail.
	 */
	public function get_custom_field_id( $field_label ) {

		$result = $this->get_api_response( 'custom_fields' );

		if ( $result['status'] === 'success' && isset( $result['response']['custom_fields'] ) ) {

			$fields = wp_list_pluck( $result['response']['custom_fields'], 'label', 'id' );

			if ( $key = array_search( $field_label, $fields ) ) {

				return (int) $key;

			} else {

				return false;
			}

		} else {

			return false;
		}
	}

	/**
	 * Creates a custom field.
	 *
	 * @since 1.1.0
	 *
	 * @param string $field_label Field label.
	 *
	 * @return int|false Field ID if success, false if fail.
	 */
	public function create_custom_field( $field_label ) {

		$result = $this->make_request( 'custom_fields', 'POST', array(
			'label' => $field_label,
		) );

		if ( $result['status'] === 'success' ) {

			return (int) $result['response']['id'];

		} else {

			return false;
		}
	}

	/**
	 * Do a remote request and retrieve the data.
	 *
	 * @since 1.0.0
	 * @access private
	 *
	 * @param string $path Part of URL.
	 * @param array $data Optional data to send through.
	 *
	 * @return array
	 */
	private function get_api_response( $path = '', $data = array() ) {

		$args = array_merge( $data, array(
			'api_key'    => $this->api_key,
			'api_secret' => $this->api_secret,
		) );

		$api_path = $this->api_url_base . $this->api_version;
		$url      = add_query_arg( $args, path_join( $api_path, $path ) );

		$response = wp_remote_get(
			$url,
			array(
				'timeout'         => 10,
				'Accept-Encoding' => 'gzip',
				'sslverify'       => false,
			)
		);

		if ( is_wp_error( $response ) ) {

			return array(
				'error' => $response->get_error_message(),
			);

		} else {

			// Maybe inflate response body.
			// @see https://wordpress.stackexchange.com/questions/10088/how-do-i-troubleshoot-responses-with-wp-http-api
			$inflate = @gzinflate( $response['body'] );
			if ( $inflate ) {
				$response['body'] = $inflate;
			}

			$body = wp_remote_retrieve_body( $response );
			$data = json_decode( $body, true );
		}

		return $data;
	}

	/**
	 * Make a request to the ConvertKit API without returning data.
	 *
	 * @since 1.0.0
	 * @access private
	 *
	 * @param string $request Request string.
	 * @param string $method HTTP Method.
	 * @param array $args Request arguments.
	 *
	 * @return array Response array.
	 */
	private function make_request( $request, $method = 'GET', $args = array() ) {

		$args = array_merge( $args, array(
			'api_key'    => $this->api_key,
			'api_secret' => $this->api_secret,
		) );

		$api_path = $this->api_url_base . $this->api_version;
		$url      = path_join( $api_path, $request );

		$headers = array(
			'Content-Type' => 'application/json; charset=utf-8',
		);

		$settings = array(
			'headers' => $headers,
			'method'  => $method,
			'body'    => wp_json_encode( $args ),
		);

		$result = wp_remote_request( $url, $settings );

		if ( is_wp_error( $result ) ) {

			return array(
				'status'   => 'failed',
				'message'  => __( 'API Response failed for unknown reasons.', 'learndash-convertkit' ),
				'response' => json_decode( wp_remote_retrieve_body( $result ), true ),
			);

		} elseif ( isset( $result['response']['code'] ) &&
		           ( $result['response']['code'] === 200 || $result['response']['code'] === 201 )
		) {

			if ( isset( $result['body'] ) ) {

				return array(
					'status'   => 'success',
					'response' => json_decode( wp_remote_retrieve_body( $result ), true ),
				);

			} else {

				return array(
					'status'   => 'failed',
					'message'  => __( 'Response from API, but no body.', 'learndash-convertkit' ),
					'response' => json_decode( wp_remote_retrieve_body( $result ), true ),
				);
			}

		} else {

			return array(
				'status'   => 'failed',
				'message'  => sprintf(
				/* translators: %d is the response code from an API call and %s is the response message */
					__( 'API Response failed with response code of %d and a message of: %s', 'learndash-convertkit' ),
					$result['response']['code'],
					$result['response']['message']
				),
				'response' => json_decode( wp_remote_retrieve_body( $result ), true ),
			);
		}
	}
}

/**
 * Gets the ConvertKit API instance.
 *
 * @since 1.1.0
 *
 * @return LD_ConvertKit_API
 */
function LD_ConvertKit_API() {

	return LearnDash_ConvertKit()->convertkit_api;
}