<?php

namespace Uncanny_Automator_Pro;

/**
 * Class MP_ADDUSERMEMBERSHIP
 * @package Uncanny_Automator_Pro
 */
class MP_ADDUSERMEMBERSHIP {

	/**
	 * Integration code
	 * @var string
	 */
	public static $integration = 'MP';

	private $action_code;
	private $action_meta;

	/**
	 * Set up Automator action constructor.
	 */
	public function __construct() {
		$this->action_code        = 'MPADDUSERMEMBERSHIP';
		$this->action_meta        = 'MPUSERMEMBERSHIP';
		$this->define_action();
	}

	/**
	 * Define and register the action by pushing it into the Automator object
	 */
	public function define_action() {

		global $uncanny_automator;

		$taxn_status = [
			\MeprTransaction::$complete_str => __( 'Complete', 'uncanny-automator' ),
			\MeprTransaction::$pending_str  => __( 'Pending', 'uncanny-automator' ),
			\MeprTransaction::$failed_str   => __( 'Failed', 'uncanny-automator' ),
			\MeprTransaction::$refunded_str => __( 'Refunded', 'uncanny-automator' ),
		];
		$mepr_options = \MeprOptions::fetch();

		$pms      = array_keys( $mepr_options->integrations );
		$gateways = [ 'manuel' => 'Manual' ];
		foreach ( $pms as $pm_id ) {
			$obj = $mepr_options->payment_method( $pm_id );
			if ( $obj instanceof \MeprBaseRealGateway ) {
				$gateways[ $obj->id ] = sprintf( __( '%1$s (%2$s)', 'uncanny-automator' ), $obj->label, $obj->name );
			}
		}

		$action = array(
			'author'             => $uncanny_automator->get_author_name(),
			'support_link'       => $uncanny_automator->get_author_support_link(),
			'integration'        => self::$integration,
			'code'               => $this->action_code,
			'sentence'           => sprintf( __( 'Add user to {{a membership:%1$s}}', 'uncanny-automator' ), $this->action_meta ),
			'select_option_name' => __( 'Add user to {{a membership}}', 'uncanny-automator' ),
			'priority'           => 10,
			'accepted_args'      => 1,
			'execution_function' => array( $this, 'add_membership' ),
			'options_group'      => [
				$this->action_meta => [
					$uncanny_automator->options->all_memberpress_products( __( 'Select a Membership', 'uncanny-automator' ), $this->action_meta, ['uo_include_any' => false] ),
					$uncanny_automator->options->text_field( 'SUBTOTAL', __( 'Sub-Total:', 'uncanny-automator' ), false, 'text', '0.0', true ),
					$uncanny_automator->options->text_field( 'TAXAMOUNT', __( 'Tax Amount:', 'uncanny-automator' ), false, 'text', '0.0', true ),
					$uncanny_automator->options->text_field( 'TAXRATE', __( 'Tax Rate:', 'uncanny-automator' ), false, 'text', '', true ),
					$uncanny_automator->options->select_field( 'STATUS', __( 'Status:', 'uncanny-automator' ), $taxn_status ),
					$uncanny_automator->options->select_field( 'GATEWAY', __( 'Gateway:', 'uncanny-automator' ), $gateways ),
					$uncanny_automator->options->text_field( 'EXPIRATIONDATE', __( 'Expiration Date:', 'uncanny-automator' ), false, 'text', '', false, "Leave empty for 'lifetime', type date in following format 'YYYY-MM-DD' for custom expiry" ),
				],
			],
		);

		$uncanny_automator->register_action( $action );
	}

	/**
	 * @param $user_id
	 * @param $action_data
	 * @param $recipe_id
	 */
	public function add_membership( $user_id, $action_data, $recipe_id ) {

		global $uncanny_automator;
		$product_id      = $action_data['meta'][ $this->action_meta ];
		$sub_total       = $action_data['meta']['SUBTOTAL'];
		$tax_amount      = $action_data['meta']['TAXAMOUNT'];
		$tax_rate        = $action_data['meta']['TAXRATE'];
		$tnx_status      = $action_data['meta']['STATUS'];
		$gateway         = $action_data['meta']['GATEWAY'];
		$expiration_date = $action_data['meta']['EXPIRATIONDATE'];
		
		$txn  = new \MeprTransaction();
		$user = new \MeprUser();
		$user->load_user_data_by_id( $user_id );
		
		$txn->trans_num  = uniqid( 'ua-mp-' );
		$txn->user_id    = $user->ID;
		$txn->product_id = sanitize_key( $product_id );
		
		$txn->amount     = (float) $sub_total;
		$txn->tax_amount = (float) $tax_amount;
		$txn->total      = ( (float) $sub_total + (float) $tax_amount );
		$txn->tax_rate   = (float) $tax_rate;
		$txn->status     = sanitize_text_field( $tnx_status );
		$txn->gateway    = sanitize_text_field( $gateway );
		$txn->created_at = \MeprUtils::ts_to_mysql_date( time() );
		
		if(isset($expiration_date) && ($expiration_date == '' || is_null($expiration_date))) {
			$txn->expires_at = \MeprUtils::db_lifetime();
		}
		else {
			$txn->expires_at = \MeprUtils::ts_to_mysql_date(strtotime($expiration_date), 'Y-m-d 23:59:59');
		}
		
		$txn->store();
		
		if ( $txn->status == \MeprTransaction::$complete_str ) {
			\MeprEvent::record( 'transaction-completed', $txn );
			
			// This is a recurring payment
			if ( ( $sub = $txn->subscription() ) && $sub->txn_count > 1 ) {
				\MeprEvent::record( 'recurring-transaction-completed',
					$txn );
			} elseif ( ! $sub ) {
				\MeprEvent::record( 'non-recurring-transaction-completed',
					$txn );
			}
		}
		
		$uncanny_automator->complete_action( $user_id, $action_data, $recipe_id );
	}
}
