<?php

namespace uncanny_automator;

/**
 * Class Development_Ready_functions
 * @package uncanny_automator
 */
class Automator_Functions {


	/**
	 * Composite Class of integration, trigger, action, and closure registration functions
	 *
	 * @since    1.0.0
	 * @access   public
	 */
	public $register = [];

	/**
	 * Collection of all recipe types
	 *
	 * @since    2.0.0
	 * @access   public
	 */
	public $recipe_types = [];

	/**
	 * Collection of all integrations
	 *
	 * @since    1.0.0
	 * @access   public
	 */
	public $integrations = [];

	/**
	 * Collection of all triggers
	 *
	 * @since    1.0.0
	 * @access   public
	 */
	public $triggers = [];

	/**
	 * Collection of all actions
	 *
	 * @since    1.0.0
	 * @access   public
	 */
	public $actions = [];

	/**
	 * Collection of all closures
	 *
	 * @since    1.0.0
	 * @access   public
	 */
	public $closures = [];

	/**
	 * Triggers and actions for each recipe with data
	 *
	 * @since    1.0.0
	 * @access   public
	 */
	public $recipes_data = [];

	/**
	 * Collection of all localized strings
	 *
	 * @since    1.0.0
	 * @access   public
	 */
	public $i18n = [];

	/**
	 * Composite Class of pre-defined Automator options
	 *
	 * @since    1.0.0
	 * @access   public
	 */
	public $options;

	/**
	 * Composite Class of pre-defined Automator utilities
	 *
	 * @since    1.0.0
	 * @access   public
	 */
	public $utilities;

	/**
	 * Composite Class of data collection functions
	 *
	 * @since    1.0.0
	 * @access   public
	 */
	public $get;

	/**
	 * Composite Class of pre-defined Automator tokens
	 *
	 * @since    1.0.0
	 * @access   public
	 */
	public $tokens;

	/**
	 * Composite Class that checks plugin status
	 *
	 * @since    1.0.0
	 * @access   public
	 */
	public $plugin_status;

	/**
	 * Composite Class that returns common error messages
	 *
	 * @since    1.0.0
	 * @access   public
	 */
	public $error_message;

	/**
	 * Composite Class that returns an input that needs to have tokens replaced
	 *
	 * @since    1.0.0
	 * @access   public
	 */
	public $parse;

	/**
	 * Collection of all Automator Email Variables
	 *
	 * @since    1.0.0
	 * @access   public
	 */
	public $defined_variables;

	/**
	 * Initializes all development helper classes and variables via class composition
	 */
	public function __construct() {

		// Automator integration, trigger, action and closure registration
		require_once $this->get_composite_class_dir() . 'automator-registration.php';
		$this->register = new Automator_Registration();

		// Load pre-defined options for triggers, actions, and closures
		require_once $this->get_composite_class_dir() . 'automator-options.php';
		$this->options = new Automator_Options();

		// Load plugin status checks
		require_once $this->get_composite_class_dir() . 'automator-integrations-status.php';
		$this->plugin_status = new Automator_Integrations_Status();

		// Load plugin status checks
		require_once $this->get_composite_class_dir() . 'automator-error-messages.php';
		$this->error_message = new Automator_Error_Messages();

		// Load plugin status checks
		require_once $this->get_composite_class_dir() . 'automator-input-parser.php';
		$this->parse = new Automator_Input_Parser();

		// Load plugin translated strings
		require_once $this->get_composite_class_dir() . 'automator-translations.php';
		$this->i18n = new Automator_Translations();

		// Load plugin translated strings
		require_once $this->get_composite_class_dir() . 'automator-utilities.php';
		$this->utilities = new Automator_Utilities();

		// Load plugin translated strings
		require_once $this->get_composite_class_dir() . 'automator-get-data.php';
		$this->get = new Automator_Get_Data();
	}

	/**
	 * Get the directory for composite classes
	 *
	 * return string
	 */
	public function get_composite_class_dir() {
		return dirname( UAP_AUTOMATOR_FILE_ ) . '/src/core/mu-classes/composite-classes/';
	}

	/**
	 * Returns a recipe types for automator
	 *
	 * @return array
	 */
	public function get_recipe_types() {
		$recipe_types = apply_filters( 'uap_recipe_types', $this->recipe_types );

		return $recipe_types;
	}

	/**
	 * Returns a filtered set on automator integrations
	 *
	 * @return array
	 */
	public function get_integrations() {
		$integrations = apply_filters( 'uap_integrations', $this->integrations );

		return $integrations;
	}

	/**
	 * Returns a filtered set on automator triggers
	 *
	 * @return array
	 */
	public function get_triggers() {
		$triggers = apply_filters( 'uap_triggers', $this->triggers );

		return $triggers;
	}

	/**
	 * Returns a filtered set on automator actions
	 *
	 * @return array
	 */
	public function get_actions() {
		$uap_actions = apply_filters( 'uap_actions', $this->actions );

		return $uap_actions;
	}

	/**
	 * Returns a filtered set on automator closures
	 *
	 * @return array
	 */
	public function get_closures() {
		$uap_actions = apply_filters( 'uap_closures', $this->closures );

		return $uap_actions;
	}

	/**
	 * @param string $code
	 *
	 * @return mixed
	 */
	public function get_author_name( $code = '' ) {
		if ( ! empty( $code ) ) {
			$code   = strtolower( $code );
			$filter = "automator_{$code}_author_name";
		} else {
			$filter = 'automator_author_name';
		}
		$author_name = apply_filters( $filter, 'Uncanny Owl' );

		return $author_name;
	}

	/**
	 * @param string $code
	 *
	 * @return mixed
	 */
	public function get_author_support_link( $code = '' ) {
		if ( ! empty( $code ) ) {
			$code   = strtolower( $code );
			$filter = "automator_{$code}_author_support_link";
		} else {
			$filter = 'automator_author_support_link';
		}

		$author_support_link = apply_filters( $filter, 'https://www.uncannyowl.com/' );

		return $author_support_link;
	}

	/**
	 * Register a new recipe type and creates a type if defined and the type does not exist
	 *
	 * @param null $recipe_type
	 * @param array $recipe_details
	 *
	 * @return null|                 |true
	 */
	public function register_recipe_type( $recipe_type = null, $recipe_details = [] ) {

		return $this->register->recipe_type( $recipe_type, $recipe_details );
	}

	/**
	 * Register a new trigger and creates a type if defined and the type does not exist
	 *
	 * @param $trigger          null||array
	 * @param $integration_code null|string
	 * @param $integration      null||array
	 *
	 * @return null|                 |true
	 */
	public function register_trigger( $trigger = null, $integration_code = null, $integration = null ) {
		return $this->register->trigger( $trigger, $integration_code, $integration );
	}

	/**
	 * Register a new uap action and creates a type if defined and the type does not exist
	 *
	 * @param $uap_action       null||array
	 * @param $integration_code null|string
	 * @param $integration      null||array
	 *
	 * @return null|                 |true
	 */
	public function register_action( $uap_action = null, $integration_code = null, $integration = null ) {
		return $this->register->action( $uap_action, $integration_code, $integration );
	}

	/**
	 * Registers a new closure and creates a type if defined and the type does not exist
	 *
	 * @param $closure          null||array
	 * @param $integration_code null|string
	 * @param $integration      null||array
	 *
	 * @return null|                 |true
	 */
	public function register_closure( $closure = null, $integration_code = null, $integration = null ) {
		return $this->register->closure( $closure, $integration_code, $integration );
	}

	/**
	 * Add a new integration
	 *
	 * @param $integration_code null||string
	 * @param $integration      null||array
	 *
	 * @return null|                 |bool
	 */
	public function register_integration( $integration_code = null, $integration = null ) {
		return $this->register->integration( $integration_code, $integration );
	}

	/**
	 * Get data for all recipe objects
	 *
	 * @param null $force_new_data_load
	 * @param null $recipe_id
	 * @param null $match_trigger_code
	 *
	 * @return array
	 */
	public function get_recipes_data( $force_new_data_load = null, $recipe_id = null, $match_trigger_code = null ) {
		if ( null === $force_new_data_load ) {
			if ( ! empty( $this->recipes_data ) ) {
				return $this->recipes_data;
			}
		}

		//$performance_setting = get_option( 'uap_automator_performance_version', 0 );
		//Always enabled
		$performance_setting = 1;
		//Force performance version
		if ( 1 === absint( $performance_setting ) && is_null( $recipe_id ) ) {
			return $this->get_recipes_data_performance();
		}

		$args = array(
			'numberposts' => 9999,
			'post_type'   => 'uo-recipe',
			'post_status' => 'any',
		);

		if ( is_numeric( $recipe_id ) && ! is_null( $recipe_id ) ) {
			$args['post__in'] = [ $recipe_id ];
			$recipe           = [];
			$recipes          = get_posts( $args );
			//$recipes = $wpdb->get_results( $wpdb->prepare( "SELECT ID, post_status FROM $wpdb->posts WHERE post_type = %s AND post_status IN ( 'publish', 'draft' ) AND ID = %d", 'uo-recipe', $recipe_id ) );
			if ( $recipes ) {
				foreach ( $recipes as $key => $recipe_d ) {

					$recipe_id                 = $recipe_d->ID;
					$completed_by_current_user = $this->is_recipe_completed( $recipe_id );

					$recipe[ $key ]['ID']          = $recipe_id;
					$recipe[ $key ]['post_status'] = $recipe_d->post_status;
					$recipe[ $key ]['recipe_type'] = $this->utilities->get_recipe_type( $recipe_id );

					$triggers_array             = []; //= is_array( $children_array ) && key_exists( 'uo-trigger', $children_array[ $recipe_id ] ) ? $children_array[ $recipe_id ]['uo-trigger'] : [];
					$triggers                   = $this->get_recipe_data( 'uo-trigger', $recipe_id, $triggers_array );
					$recipe[ $key ]['triggers'] = $triggers;

					$action_array              = []; //= is_array( $children_array ) && key_exists( 'uo-action', $children_array[ $recipe_id ] ) ? $children_array[ $recipe_id ]['uo-action'] : [];
					$actions                   = $this->get_recipe_data( 'uo-action', $recipe_id, $action_array );
					$recipe[ $key ]['actions'] = $actions;

					$closure_array              = []; //= is_array( $children_array ) && key_exists( 'uo-closure', $children_array[ $recipe_id ] ) ? $children_array[ $recipe_id ]['uo-action'] : [];
					$closures                   = $this->get_recipe_data( 'uo-closure', $recipe_id, $closure_array );
					$recipe[ $key ]['closures'] = $closures;

					$recipe[ $key ]['completed_by_current_user'] = $completed_by_current_user;
				}
			}

			return $recipe;
		} else {

			global $uncanny_automator;
			/*if ( ! is_null( $match_trigger_code ) ) {
				$args['meta_key']   = 'code';
				$args['meta_value'] = $match_trigger_code;
				$args['compare']    = 'LIKE';
			}*/
			$recipes = get_posts( $args );
			if ( $recipes ) {
				foreach ( $recipes as $key => $recipe ) {

					$recipe_id = $recipe->ID;

					$triggers_array                         = []; //= is_array( $children_array ) && ! empty( $children_array ) && key_exists( 'uo-trigger', $children_array[ $recipe_id ] ) ? $children_array[ $recipe_id ]['uo-trigger'] : [];
					$triggers                               = $this->get_recipe_data( 'uo-trigger', $recipe_id, $triggers_array, $match_trigger_code );
					$this->recipes_data[ $key ]['triggers'] = $triggers;
					//No recipes matched trigger code.. Continue
					if ( empty( $triggers ) ) {
						//unset( $this->recipes_data[ $key ] );
						continue;
					}

					$this->recipes_data[ $key ]['ID']          = $recipe_id;
					$this->recipes_data[ $key ]['post_status'] = $recipe->post_status;
					$this->recipes_data[ $key ]['recipe_type'] = $uncanny_automator->utilities->get_recipe_type( $recipe_id );


					$action_array                          = []; //= is_array( $children_array ) && ! empty( $children_array ) && key_exists( 'uo-action', $children_array[ $recipe_id ] ) ? $children_array[ $recipe_id ]['uo-action'] : [];
					$actions                               = $this->get_recipe_data( 'uo-action', $recipe_id, $action_array );
					$this->recipes_data[ $key ]['actions'] = $actions;

					$closure_array                          = []; //= is_array( $children_array ) && ! empty( $children_array ) && key_exists( 'uo-closure', $children_array[ $recipe_id ] ) ? $children_array[ $recipe_id ]['uo-closure'] : [];
					$closures                               = $this->get_recipe_data( 'uo-closure', $recipe_id, $closure_array );
					$this->recipes_data[ $key ]['closures'] = $closures;

					$this->recipes_data[ $key ]['completed_by_current_user'] = $this->is_recipe_completed( $recipe_id );
				}
			}

			return $this->recipes_data;
		}
	}

	/**
	 * @param null $match_trigger_code
	 *
	 * @return array
	 */
	public function get_recipes_data_performance() {
		global $uncanny_automator;
		global $wpdb;
		//if ( is_admin() || DOING_AJAX || isset( $_REQUEST['doing_rest'] ) ) {
		$qry = $wpdb->prepare( "SELECT ID, post_title, post_type, post_status, post_parent FROM $wpdb->posts WHERE post_type = %s ORDER BY ID DESC LIMIT 0, 99999", 'uo-recipe' );
		/*} else {
			if ( is_user_logged_in() ) {
				$recipe_type = 'user';
			} else {
				$recipe_type = 'anonymous';
			}
			$qry = $wpdb->prepare( "SELECT
									p.ID, p.post_title, p.post_type, p.post_status, p.post_parent
									FROM $wpdb->posts p
									LEFT JOIN $wpdb->postmeta pm
									ON pm.post_id = p.ID
									WHERE 1=1
									AND p.post_type = %s
									#AND p.post_status = 'publish'
									AND pm.meta_key = 'uap_recipe_type'
									AND pm.meta_value = $recipe_type
									ORDER BY p.ID DESC
									LIMIT 0, 99999", 'uo-recipe' );
		}*/

		//echo $qry;
		$recipes = $wpdb->get_results( $qry );
		//Extract Recipe IDs
		$recipe_ids = array_column( (array) $recipes, 'ID' );
		//Collective array of recipes triggers, actions, closures
		$recipe_data = $this->pre_fetch_recipe_metas( $recipes );
		//Collective array of users recipes completed status
		$recipes_completed = $this->are_recipes_completed( null, $recipe_ids );
		$recipes_completed = empty( $recipes_completed ) ? [] : $recipes_completed;
		if ( $recipes ) {
			$key = 0;
			foreach ( $recipes as $recipe ) {
				$recipe_id = $recipe->ID;
				if ( key_exists( $recipe_id, $recipe_data ) && is_array( $recipe_data[ $recipe_id ] ) && key_exists( 'triggers', $recipe_data[ $recipe_id ] ) ) {
					if ( $recipe_data[ $recipe_id ]['triggers'] ) {
						//Grab tokens for each of trigger
						foreach ( $recipe_data[ $recipe_id ]['triggers'] as $t_id => $tr ) {
							$tokens                                                   = $this->get->recipe_trigger_tokens( $tr['meta'], $recipe_id );
							$recipe_data[ $recipe_id ]['triggers'][ $t_id ]['tokens'] = $tokens;
						}
					}
					$triggers = $recipe_data[ $recipe_id ]['triggers']; //$this->get_recipe_data( 'uo-trigger', $recipe_id, $triggers_array );
				} else {
					$triggers = [];
				}
				/*if ( empty( $triggers ) ) {
					continue;
				}*/
				$this->recipes_data[ $key ]['ID']          = $recipe_id;
				$this->recipes_data[ $key ]['post_status'] = $recipe->post_status;
				$this->recipes_data[ $key ]['recipe_type'] = $uncanny_automator->utilities->get_recipe_type( $recipe_id );

				$this->recipes_data[ $key ]['triggers'] = $triggers;

				if ( key_exists( $recipe_id, $recipe_data ) && is_array( $recipe_data[ $recipe_id ] ) && key_exists( 'actions', $recipe_data[ $recipe_id ] ) ) {
					$actions = $recipe_data[ $recipe_id ]['actions']; //$this->get_recipe_data( 'uo-action', $recipe_id, $action_array );
				} else {
					$actions = [];
				}
				$this->recipes_data[ $key ]['actions'] = $actions;

				if ( key_exists( $recipe_id, $recipe_data ) && is_array( $recipe_data[ $recipe_id ] ) && key_exists( 'closures', $recipe_data[ $recipe_id ] ) ) {
					$closures = $recipe_data[ $recipe_id ]['closures']; //$this->get_recipe_data( 'uo-closure', $recipe_id, $closure_array );
				} else {
					$closures = [];
				}
				$this->recipes_data[ $key ]['closures'] = $closures;

				$this->recipes_data[ $key ]['completed_by_current_user'] = key_exists( $recipe_id, $recipes_completed ) ? $recipes_completed[ $recipe_id ] : false;
				$key ++;
			}
		}

		return $this->recipes_data;
	}

	/**
	 * @param array $recipes
	 *
	 * @return array
	 */
	public function pre_fetch_recipe_metas( $recipes = [] ) {
		$metas    = [];
		$triggers = [];
		$actions  = [];
		$closures = [];
		if ( ! empty( $recipes ) ) {

			global $wpdb;
			//////////////////////
			//////////////////////
			//////////////////////
			/////Fetch uo-trigger, uo-action, uo-closure
			$recipe_children = $wpdb->get_results( "SELECT ID, post_status, post_type
													FROM $wpdb->posts
													WHERE post_parent IN (SELECT ID
													FROM $wpdb->posts
													WHERE post_type = 'uo-recipe')" );

			if ( $recipe_children ) {
				foreach ( $recipe_children as $p ) {
					$ID  = $p->ID;
					$p_t = $p->post_type;
					$p_s = $p->post_status;
					switch ( $p_t ) {
						case 'uo-trigger':
							$triggers[ $ID ] = [ 'ID' => $ID, 'post_status' => $p_s, ];
							break;
						case 'uo-action':
							$actions[ $ID ] = [ 'ID' => $ID, 'post_status' => $p_s, ];
							break;
						case 'uo-closure':
							$closures[ $ID ] = [ 'ID' => $ID, 'post_status' => $p_s, ];
							break;
					}

				}
			}

			///END
			//////////////////////
			//////////////////////

			//////////////////////
			//////////////////////
			//////////////////////
			/////Fetch metas for uo-trigger, uo-action, uo-closure
			$q             = $wpdb->prepare( "SELECT pm.post_id, pm.meta_key, pm.meta_value, p.post_parent, p.post_type
														FROM $wpdb->postmeta pm
														LEFT JOIN $wpdb->posts p
														ON p.ID = pm.post_id
														WHERE pm.post_id IN (SELECT ID
														FROM $wpdb->posts
														WHERE post_parent IN (SELECT ID
														FROM $wpdb->posts
														WHERE post_type = %s))", 'uo-recipe' );
			$related_metas = $wpdb->get_results( $q );

			if ( $related_metas ) {
				foreach ( $related_metas as $p ) {
					$ID  = $p->post_id;
					$m_k = $p->meta_key;
					$m_v = $p->meta_value;
					if ( key_exists( $ID, $triggers ) ) {
						$triggers[ $ID ]['meta'][ $m_k ] = $m_v;
					} elseif ( key_exists( $ID, $actions ) ) {
						$actions[ $ID ]['meta'][ $m_k ] = $m_v;
					} elseif ( key_exists( $ID, $closures ) ) {
						$closures[ $ID ]['meta'][ $m_k ] = $m_v;
					}
				}
			}
			//Fix missing metas!
			if ( $triggers ) {
				foreach ( $triggers as $trigger_ID => $array ) {
					if ( ! key_exists( 'meta', $array ) ) {
						$triggers[ $trigger_ID ]['meta'] = [ 'code' => '' ];
					} elseif ( key_exists( 'meta', $array ) ) {
						//Attempt to return Trigger ID for magic button
						foreach ( $array['meta'] as $mk => $mv ) {
							if ( 'code' === (string) trim( $mk ) && 'WPMAGICBUTTON' === (string) trim( $mv ) ) {
								$triggers[ $trigger_ID ]['meta']['WPMAGICBUTTON'] = $trigger_ID;
							}
						}
					}
				}
			}

			///END
			//////////////////////
			//////////////////////


			/////Build old recipe array style
			foreach ( $related_metas as $r ) {
				$recipe_id     = absint( $r->post_parent );
				$non_recipe_id = absint( $r->post_id );
				switch ( $r->post_type ) {
					case 'uo-trigger':
						if ( key_exists( $non_recipe_id, $triggers ) ) {
							$metas[ $recipe_id ]['triggers'][] = $triggers[ $non_recipe_id ];
							unset( $triggers[ $non_recipe_id ] );
						}
						break;
					case 'uo-action':
						if ( key_exists( $non_recipe_id, $actions ) ) {
							$metas[ $recipe_id ]['actions'][] = $actions[ $non_recipe_id ];
							unset( $actions[ $non_recipe_id ] );
						}
						break;
					case 'uo-closure':
						if ( key_exists( $non_recipe_id, $closures ) ) {
							$metas[ $recipe_id ]['closures'][] = $closures[ $non_recipe_id ];
							unset( $closures[ $non_recipe_id ] );
						}
						break;
				}
			}
		}


		return $metas;
	}

	/**
	 * Check if the recipe was completed
	 *
	 * @param $user_id   null||int
	 * @param array $recipe_ids
	 *
	 * @return array
	 */
	public function are_recipes_completed( $user_id = null, $recipe_ids = [] ) {

		if ( empty( $recipe_ids ) ) {
			Utilities::log( 'ERROR: You are trying to check if a recipe is completed without providing a recipe_id.', 'is_recipe_completed ERROR', false, 'uap-errors' );

			return null;
		}

		// Set user ID
		if ( null === $user_id ) {
			$user_id = get_current_user_id();
		}

		// No user id is aviable.
		if ( 0 === $user_id ) {
			Utilities::log( 'ERROR: You are trying to check if a recipe is completed when a there is no logged in user.', 'is_recipe_completed ERROR', false, 'uap-errors' );

			return null;
		}

		$completed = [];
		global $wpdb;
		$table_name = $wpdb->prefix . 'uap_recipe_log';
		if ( ! empty( $recipe_ids ) ) {
			$results = $wpdb->get_results( "SELECT COUNT(completed) AS completed, automator_recipe_id FROM $table_name WHERE user_id = $user_id AND automator_recipe_id IN (" . join( ',', $recipe_ids ) . ") AND completed = 1 GROUP BY automator_recipe_id" );

			if ( $results ) {
				foreach ( $recipe_ids as $recipe_id ) {
					$complete = 0;
					$found    = false;
					foreach ( $results as $r ) {
						if ( $recipe_id === $r->automator_recipe_id ) {
							$found    = true;
							$complete = $r->completed;
							break;
						} else {
							$found = false;
						}
					}

					if ( $found ) {
						$completed[ $recipe_id ] = $complete;
					} else {
						$completed[ $recipe_id ] = 0;
					}
				}
			} else {
				//Fallback to mark every recipe incomplete
				foreach ( $recipe_ids as $recipe_id ) {
					$completed[ $recipe_id ] = 0;
				}
			}
		}

		return $this->utilities->recipes_number_times_completed( $recipe_ids, $completed );

	}

	/**
	 * Check if the recipe was completed
	 *
	 * @param $recipe_id null||int
	 * @param $user_id   null||int
	 * @param bool $is_anonymous v2.0
	 *
	 * @return null|bool
	 */
	public function is_recipe_completed( $recipe_id = null, $user_id = null, $is_anonymous = false ) {

		if ( null === $recipe_id || ! is_numeric( $recipe_id ) ) {
			Utilities::log( 'ERROR: You are trying to check if a recipe is completed without providing a recipe_id.', 'is_recipe_completed ERROR', false, 'uap-errors' );

			return null;
		}

		if ( $is_anonymous && 0 === $user_id ) {
			return false;
		}

		// Set user ID
		if ( is_null( $user_id ) ) {
			$user_id = get_current_user_id();
		}

		// No user id is aviable.
		/*if ( 0 === $user_id ) {
			Utilities::log( 'ERROR: You are trying to check if a recipe is completed when a there is no logged in user.', 'is_recipe_completed ERROR', false, 'uap-errors' );

			return null;
		}*/

		global $wpdb;
		$table_name = $wpdb->prefix . 'uap_recipe_log';
		$results    = $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(completed) FROM $table_name WHERE user_id = %d AND automator_recipe_id = %d AND completed = 1", $user_id, $recipe_id ) );

		if ( 0 === $results ) {
			return false;
		} else {
			$results = empty( $results ) ? 0 : $results;

			return $this->utilities->recipe_number_times_completed( $recipe_id, $results );
		}
	}

	/**
	 * Complete a recipe
	 *
	 * @param $recipe_id     null||int
	 * @param $user_id       null||int
	 * @param $recipe_log_id null||int
	 *
	 * @param array $args
	 *
	 * @return null|true
	 */
	public function complete_recipe( $recipe_id = null, $user_id = null, $recipe_log_id = null, $args = [] ) {

		if ( null === $recipe_id || ! is_numeric( $recipe_id ) ) {
			Utilities::log( 'ERROR: You are trying to completed a recipe without providing a recipe_id', 'complete_recipe ERROR', false, 'uap-errors' );

			return null;
		}

		if ( ( is_array( $args ) && key_exists( 'do-nothing', $args ) ) ) {
			$completed  = 9;
			$run_number = 1;
		} else {
			$completed  = 1;
			$run_number = $this->get->next_run_number( $recipe_id, $user_id, true );
		}

		do_action( 'uap_before_recipe_completed', $recipe_id, $user_id, $recipe_log_id, $args );

		/*Utilities::log( [
			'$recipe_id'     => $recipe_id,
			'$user_id'       => $user_id,
			'$recipe_log_id' => $recipe_log_id,
			'$completed'     => $completed,
			'$args'          => $args
		], '', true, 'step-final' );*/

		global $wpdb;
		$table_name = $wpdb->prefix . 'uap_recipe_log';
		if ( null === $recipe_log_id ) {
			$wpdb->insert( $table_name,
				array(
					'date_time'           => date( 'Y-m-d H:i:s', current_time( 'timestamp' ) ),
					'user_id'             => $user_id,
					'automator_recipe_id' => $recipe_id,
					'completed'           => $completed,
					'run_number'          => $run_number,
				), array(
					'%s',
					'%d',
					'%d',
					'%d',
					'%d',
				) );

			$recipe_log_id = $wpdb->insert_id;

		} else {
			$wpdb->update( $table_name,
				array(
					'date_time'  => date( 'Y-m-d H:i:s', current_time( 'timestamp' ) ),
					'completed'  => $completed,
					'run_number' => $run_number
				),
				array(
					'ID'                  => $recipe_log_id,
					'automator_recipe_id' => $recipe_id,
					'user_id'             => $user_id,
				),
				array(
					'%s',
					'%d',
					'%d',
				),
				array(
					'%d',
					'%d',
					'%d',
				) );
		}

		// If actions error occured, change the recipe status to 2
		$action_table_name = $wpdb->prefix . 'uap_action_log';
		$q                 = "SELECT error_message, completed FROM {$action_table_name} WHERE error_message != '' AND automator_recipe_log_id = $recipe_log_id";
		$maybe_error       = $wpdb->get_row( $q );
		/*Utilities::log( [
			'$q'           => $q,
			'$maybe_error' => $maybe_error,
		], '', true, 'step-final' );*/

		if ( ! empty( $maybe_error ) ) {
			$skip     = false;
			$message  = $maybe_error->error_message;
			$complete = $maybe_error->completed;

			if ( strpos( $message, 'Existing user found matching' ) || strpos( $message, 'User not found matching' ) || strpos( $message, 'User found matching' ) ) {
				$skip = true;
			} elseif ( strpos( $message, 'New user created' ) || strpos( $message, 'Create new user failed' ) ) {
				$skip = true;
			} elseif ( 9 === (int) $complete ) {
				$skip = true;
			}

			if ( ! $skip ) {
				$wpdb->update( $table_name,
					array(
						'completed' => 9 === (int) $completed ? 9 : $complete,
					),
					array(
						'ID' => $recipe_log_id,
					),
					array( '%d', ),
					array( '%d', ) );
			}
		}

		do_action( 'uap_recipe_completed', $recipe_id, $user_id, $recipe_log_id, $args );

		$this->complete_closures( $recipe_id, $user_id, $recipe_log_id, $args );
	}

	/**
	 * Complete all actions in recipe
	 *
	 * @param null $recipe_id
	 * @param null $user_id
	 * @param null $recipe_log_id
	 *
	 * @param array $args
	 *
	 * @return bool
	 */
	public function complete_actions( $recipe_id = null, $user_id = null, $recipe_log_id = null, $args = [] ) {

		$recipe_action_data = $this->get_recipe_data( 'uo-action', $recipe_id );

		foreach ( $recipe_action_data as $action_data ) {

			$action_code                  = $action_data['meta']['code'];
			$action_status                = $action_data['post_status'];
			$action_data['recipe_log_id'] = $recipe_log_id;
			$action_integration           = $this->get->action_integration_from_action_code( $action_code );

			if ( 1 === $this->plugin_status->get( $action_integration ) && 'publish' === $action_status ) {
				// The plugin for this action is active .. execute
				$action_execution_function = $this->get->action_execution_function_from_action_code( $action_code );

				$valid_function = true;
				if ( null === $action_execution_function ) {
					$valid_function = false;
				} elseif ( is_array( $action_execution_function ) && ! method_exists( $action_execution_function[0], $action_execution_function[1] ) ) {
					$valid_function = false;
				} elseif ( is_string( $action_execution_function ) && ! function_exists( $action_execution_function ) ) {
					$valid_function = false;
				}

				if ( ! $valid_function ) {
					global $uncanny_automator;
					$error_message = $uncanny_automator->error_message->get( 'action-function-not-exist' );
					$uncanny_automator->complete_action( $user_id, $action_data, $recipe_id, $error_message, $recipe_log_id, $args );
				} else {
					/*Utilities::log( [
						'$action_execution_function' => $action_execution_function,
						'$user_id'                   => $user_id,
						'$action_data'               => $action_data,
						'$recipe_id'                 => $recipe_id,
						'$args'                      => $args,
					], 'call_user_func_array', true, 'step3-b' );*/
					//fallback...
					$action_data['args'] = $args;
					call_user_func_array( $action_execution_function, array(
						$user_id,
						$action_data,
						$recipe_id,
						$args,
					) );
				}

			} elseif ( 0 === $this->plugin_status->get( $action_integration ) ) {
				global $uncanny_automator;
				$error_message = $uncanny_automator->error_message->get( 'action-not-active' );
				$uncanny_automator->complete_action( $user_id, $action_data, $recipe_id, $error_message, $recipe_log_id, $args );
			} elseif ( 'draft' !== $action_status ) {
				global $uncanny_automator;
				$error_message = $uncanny_automator->error_message->get( 'plugin-not-active' );
				$uncanny_automator->complete_action( $user_id, $action_data, $recipe_id, $error_message, $recipe_log_id, $args );
			}
		}

		/*Utilities::log( [
			'$recipe_id'     => $recipe_id,
			'$user_id'       => $user_id,
			'$recipe_log_id' => $recipe_log_id,
			'$args'          => $args
		], '', true, 'step4' );*/
		$this->complete_recipe( $recipe_id, $user_id, $recipe_log_id, $args );

		return true;
	}

	/**
	 * Complete all closures in recipe
	 *
	 * @param null $recipe_id
	 * @param null $user_id
	 * @param null $recipe_log_id
	 * @param array $args
	 *
	 * @return bool
	 *
	 */
	public function complete_closures( $recipe_id = null, $user_id = null, $recipe_log_id = null, $args = [] ) {

		/*if ( ! is_user_logged_in() ) {
			return false;
		}*/

		$recipe_closure_data = $this->get_recipe_data( 'uo-closure', $recipe_id );
		foreach ( $recipe_closure_data as $closure_data ) {

			$closure_code                  = $closure_data['meta']['code'];
			$closure_status                = $closure_data['post_status'];
			$closure_data['recipe_log_id'] = $recipe_log_id;
			$closure_integration           = $this->get->closure_integration_from_closure_code( $closure_code );

			if ( 1 === $this->plugin_status->get( $closure_integration ) && 'publish' === $closure_status ) {

				// The plugin for this action is active .. execute
				$closure_execution_function = $this->get->closure_execution_function_from_closure_code( $closure_code );
				call_user_func_array( $closure_execution_function, array(
					$user_id,
					$closure_data,
					$recipe_id,
					$args,
				) );
			} else {

				// The plugin for this action is NOT active
				Utilities::log( 'ERROR: You are trying to complete ' . $closure_code . ' and the plugin ' . $closure_integration . ' is not active. ', 'complete_closures ERROR', false, 'uap-errors' );
			}
		}

		do_action( 'uap_closures_completed', $recipe_id, $user_id, $args );

		return true;
	}

	/**
	 * Get saved data for recipe actions or triggers
	 *
	 * @param       $type      null||int
	 * @param       $recipe_id null||int
	 * @param array $recipe_children
	 * @param null $matched_trigger
	 *
	 * @return null|                |array
	 */
	public function get_recipe_data( $type = null, $recipe_id = null, $recipe_children = [], $matched_trigger = null ) {

		if ( null === $type ) {
			return null;
		}

		if ( ! in_array( $type, array( 'uo-trigger', 'uo-action', 'uo-closure' ) ) ) {
			return null;
		}


		if ( is_null( $recipe_id ) || ! is_numeric( $recipe_id ) ) {
			Utilities::log( 'ERROR: You are trying to get recipe data without providing a recipe_id', 'get_recipe_data ERROR', false, 'uap-errors' );

			return null;
		}

		global $wpdb;
		if ( empty( $recipe_children ) ) {
			$q = apply_filters(
				'q_get_recipe_data',
				"Select ID, post_status FROM $wpdb->posts WHERE post_parent = $recipe_id AND post_type = '{$type}'",
				$recipe_id,
				$type );

			// All the triggers associated with the recipe
			$recipe_children = $wpdb->get_results( $q, ARRAY_A );
		}
		// All data for recipe triggers
		$recipe_children_data = [];
		if ( $recipe_children ) {
			// Check each trigger for set values
			foreach ( $recipe_children as $key => $child ) {
				// Collect all meta data for this trigger
				if ( ! key_exists( 'meta', $child ) ) {
					$child_meta = get_post_custom( $child['ID'] );
				} else {
					$child_meta = $child['meta'];
				}
				// Get post custom return an array for each meta_key as there maybe more than one value per key.. we only store and need one value
				$child_meta_single = [];

				foreach ( $child_meta as $meta_key => $meta_value ) {
					$child_meta_single[ $meta_key ] = reset( $meta_value );
				}
				$code = key_exists( 'code', $child_meta_single ) ? $child_meta_single['code'] : '';

				$item_not_found = true;

				if ( 'uo-trigger' === $type ) {
					$system_triggers = $this->get_triggers();
					if ( ! empty( $system_triggers ) ) {
						foreach ( $system_triggers as $trigger ) {
							if ( $trigger['code'] === $code ) {
								$item_not_found = false;
							}
						}
					} else {
						$item_not_found = false;
					}
				}


				if ( 'uo-action' === $type ) {
					$system_actions = $this->get_actions();
					if ( ! empty( $system_actions ) ) {
						foreach ( $system_actions as $action ) {
							if ( $action['code'] === $code ) {
								$item_not_found = false;
							}
						}
					} else {
						$item_not_found = false;
					}
				}


				if ( 'uo-closure' === $type ) {
					$system_closures = $this->get_closures();
					if ( ! empty( $system_closures ) ) {
						foreach ( $system_closures as $closure ) {
							if ( $closure['code'] === $code ) {
								$item_not_found = false;
							}
						}
					} else {
						$item_not_found = false;
					}
				}


				if ( $item_not_found ) {

					$item = array(
						'ID'          => $child['ID'],
						'post_status' => 'draft',
					);

					//wp_update_post( $item );
					$wpdb->query( "UPDATE $wpdb->posts SET post_status = 'draft' WHERE ID = {$child['ID']}" );
					$child['post_status'] = 'draft';
				}

				// The trigger is create/stored automatically but may not have been saved. Delete if not saved!
				if ( empty( $child_meta ) && isset( $child['ID'] ) ) {
					//wp_delete_post( $child['ID'] );
					continue;
				}

				$recipe_children_data[ $key ]['ID']          = absint( $child['ID'] );
				$recipe_children_data[ $key ]['post_status'] = $child['post_status'];
				$recipe_children_data[ $key ]['meta']        = $child_meta_single;


				if ( 'uo-trigger' === $type ) {
					$recipe_children_data[ $key ]['tokens'] = $this->get->recipe_trigger_tokens( $child_meta_single, $recipe_id );
				}
			}
		}

		return $recipe_children_data;
	}

	/**
	 * Insert trigger for the user
	 *
	 * @param $user_id
	 * @param $trigger_id    null||int
	 * @param $recipe_id     null||int
	 * @param $completed     bool
	 * @param $recipe_log_id null||bool
	 *
	 * @return int|null
	 */
	public function insert_trigger( $user_id = null, $trigger_id = null, $recipe_id = null, $completed = false, $recipe_log_id = null ) {
		if ( null === $trigger_id ) {
			return null;
		}
		if ( is_null( $user_id ) ) {
			return null;
		}
		if ( null === $recipe_id ) {
			return null;
		}

		global $wpdb;

		$table_name = $wpdb->prefix . 'uap_trigger_log';

		$wpdb->insert( $table_name,
			array(
				'date_time'               => date( 'Y-m-d H:i:s', current_time( 'timestamp' ) ),
				'user_id'                 => $user_id,
				'automator_trigger_id'    => $trigger_id,
				'automator_recipe_id'     => $recipe_id,
				'completed'               => $completed,
				'automator_recipe_log_id' => $recipe_log_id,
			), array(
				'%s',
				'%d',
				'%d',
				'%d',
				'%s',
				'%d',
			) );

		return (int) $wpdb->insert_id;
	}

	/**
	 * Check if the trigger is completed
	 *
	 * @param $user_id       null||int
	 * @param $trigger_id    null||int
	 * @param $recipe_id     null||int
	 * @param $recipe_log_id null||int
	 *
	 * @param array $args
	 *
	 * @return null|bool
	 */
	public function is_trigger_completed( $user_id = null, $trigger_id = null, $recipe_id = null, $recipe_log_id = null, $args = [] ) {

		// Set user ID
		if ( is_null( $user_id ) ) {
			$user_id = get_current_user_id();
		}

		// No user id is available.
		/*if ( 0 === $user_id ) {
			Utilities::log( 'ERROR: You are trying to check if a trigger is completed when a there is no logged in user.', 'is_trigger_completed ERROR', false, 'uap-errors' );

			return false;
		}*/

		if ( null === $trigger_id || ! is_numeric( $trigger_id ) ) {
			Utilities::log( 'ERROR: You are trying to check if a trigger is completed without providing a trigger_id', 'is_trigger_completed ERROR', false, 'uap-errors' );

			return null;
		}

		if ( null === $recipe_id || ! is_numeric( $recipe_id ) ) {
			Utilities::log( 'ERROR: You are trying to check if a trigger is completed without providing a recipe_id', 'is_trigger_completed ERROR', false, 'uap-errors' );

			return null;
		}

		global $wpdb;
		$table_name = $wpdb->prefix . 'uap_trigger_log';
		$q          = "SELECT completed FROM $table_name WHERE user_id = %d AND automator_trigger_id = %d AND automator_recipe_id = %d AND automator_recipe_log_id = %d";
		$results    = $wpdb->get_var( $wpdb->prepare( $q, $user_id, $trigger_id, $recipe_id, $recipe_log_id ) );

		$return = false;
		if ( empty( $results ) ) {
			$return = false;
		} elseif ( ! empty( $results ) ) {
			$return = true;
		}

		return $return;
	}

	/**
	 * Update the trigger for the user
	 *
	 * @param $user_id    null||int
	 * @param $trigger_id null||int
	 * @param $recipe_id  null||int
	 * @param $ID         null||int
	 *
	 * @return null
	 */
	public function update_trigger( $user_id = null, $trigger_id = null, $recipe_id = null, $ID = null ) {

		// Set user ID
		if ( null === $user_id ) {
			$user_id = get_current_user_id();
		}

		// No user id is aviable.
		if ( 0 === $user_id ) {
			Utilities::log( 'ERROR: You are trying to update a trigger when a there is no logged in user.', 'update_trigger ERROR', false, 'uap-errors' );

			return null;
		}

		if ( null === $trigger_id || ! is_numeric( $trigger_id ) ) {
			Utilities::log( 'ERROR: You are trying to update a trigge without providing a trigger_id', 'update_trigger ERROR', false, 'uap-errors' );

			return null;
		}

		if ( null === $recipe_id || ! is_numeric( $recipe_id ) ) {
			Utilities::log( 'ERROR: You are trying to update a trigge without providing a recipe_id', 'update_trigger ERROR', false, 'uap-errors' );

			return null;
		}
	}

	/**
	 * Complete the trigger for the user
	 *
	 * @param array $args
	 *
	 * @return null
	 */
	public function complete_trigger( $args = [] ) {

		$user_id        = absint( $args['user_id'] );
		$trigger_id     = absint( $args['trigger_id'] );
		$recipe_id      = absint( $args['recipe_id'] );
		$trigger_log_id = absint( $args['get_trigger_id'] );
		$recipe_log_id  = absint( $args['recipe_log_id'] );

		// Set user ID
		if ( is_null( $user_id ) ) {
			$user_id = get_current_user_id();
		}

		if ( null === $trigger_id || ! is_numeric( $trigger_id ) ) {
			Utilities::log( 'ERROR: You are trying to complete a trigge without providing a trigger_id', 'complete_trigger ERROR', false, 'uap-errors' );

			return null;
		}

		if ( null === $recipe_id || ! is_numeric( $recipe_id ) ) {
			Utilities::log( 'ERROR: You are trying to complete a trigge without providing a recipe_id', 'complete_trigger ERROR', false, 'uap-errors' );

			return null;
		}
		// The trigger is about to be completed
		do_action( 'uap_before_trigger_completed', $user_id, $trigger_id, $recipe_id, $trigger_log_id, $args );

		$trigger_code        = get_post_meta( $trigger_id, 'code', true );
		$trigger_integration = $this->get->trigger_integration_from_trigger_code( $trigger_code );
		if ( 0 === $this->plugin_status->get( $trigger_integration ) ) {

			// The plugin for this action is NOT active
			Utilities::log( 'ERROR: You are trying to complete ' . $trigger_code . ' and the plugin ' . $trigger_integration . ' is not active. ', 'complete_trigger ERROR', false, 'uap-errors' );

			return null;
		}

		global $wpdb;

		$update = array(
			'completed' => true,
			'date_time' => date( 'Y-m-d H:i:s', current_time( 'timestamp' ) ),
		);

		$where = array(
			'user_id'              => $user_id,
			'automator_trigger_id' => $trigger_id,
			'automator_recipe_id'  => $recipe_id,
		);

		if ( null !== $trigger_log_id && is_int( $trigger_log_id ) ) {
			$where['ID'] = (int) $trigger_log_id;
		}

		if ( null !== $recipe_log_id && is_int( $recipe_log_id ) ) {
			$where['automator_recipe_log_id'] = (int) $recipe_log_id;
		}

		$update_format = array(
			'%d',
			'%s',
		);

		$where_format = array(
			'%d',
			'%d',
			'%d',
		);

		if ( ! empty( $trigger_log_id ) && is_int( $trigger_log_id ) ) {
			$where_format[] = '%d';
		}
		if ( ! empty( $recipe_log_id ) && is_int( $recipe_log_id ) ) {
			$where_format[] = '%d';
		}

		$table_name = $wpdb->prefix . 'uap_trigger_log';

		$wpdb->update( $table_name,
			$update,
			$where,
			$update_format,
			$where_format
		);

		// The trigger is now completed
		do_action( 'uap_trigger_completed', $user_id, $trigger_id, $recipe_id, $trigger_log_id, $args );

		$maybe_continue_recipe_process = true;
		//make sure that we have a User_ID selected for next step.
		$recipe_type = (string) $this->utilities->get_recipe_type( $recipe_id );
		if ( 'anonymous' === (string) $recipe_type ) {
			do_action( 'uap_before_user_action_completed', $user_id, $recipe_id, $trigger_log_id, $args );
			$user_action_result = $this->maybe_run_anonymous_recipe_user_actions( $recipe_id, $user_id, $recipe_log_id, $args );
			$user_action_status = $user_action_result['status'];
			$user_action_data   = $user_action_result['data'];
			$user_id            = key_exists( 'user_id', $user_action_data ) ? $user_action_data['user_id'] : false;

			/*Utilities::log( [
				'$user_action_result' => $user_action_result,
				'$recipe_log_id'      => $recipe_log_id,
				'$user_id'            => $user_id,
				'$recipe_id'          => $recipe_id,
				'$trigger_log_id'     => $trigger_log_id,
				'$args'               => $args
			], 'maybe_run_anonymous_recipe_user_actions', true, 'step1' );*/

			//$user_id = $this->complete_anon_user_actions( $recipe_id, $user_id, $recipe_log_id, $args );
			if ( ! $user_action_status ) {
				$maybe_continue_recipe_process = false;
				$recipe_action_data            = $this->get_recipe_data( 'uo-action', $recipe_id );
				//if ( $recipe_action_data ) {
				foreach ( $recipe_action_data as $action_data ) {
					//Create New User - Existing user found matching
					$error_message = key_exists( 'message', $user_action_data ) ? $user_action_data['message'] : __( 'There was an error completing recipe', 'uncanny-automator' );

					do_action( 'uap_before_user_action_do_nothing_completed', $user_action_result, $user_id, $action_data, $recipe_id, $trigger_log_id, $args );

					$action_data['do-nothing'] = true;

					if ( key_exists( 'error', $user_action_data ) ) {
						$action_data['complete_with_errors'] = true;
					}

					$action_data['recipe_log_id'] = $recipe_log_id;
					$args['user_action_message']  = $user_action_data['message'];

					if ( $user_id ) {
						//attempt to update anonymous user to actual user!
						$table_name = $wpdb->prefix . 'uap_recipe_log';
						$wpdb->update( $table_name, [ 'user_id' => $user_id ],
							[ 'ID' => $args['recipe_log_id'] ],
							[ '%d' ],
							[ '%d' ]
						);
					}

					$this->complete_action( $user_id, $action_data, $recipe_id, $error_message, $recipe_log_id, $args ); //Complete action with errors!

					do_action( 'uap_after_user_action_do_nothing_completed', $user_action_result, $user_id, $action_data, $recipe_id, $trigger_log_id, $args );

					$args['do-nothing'] = true;

					$this->complete_recipe( $recipe_id, $user_id, $recipe_log_id, $args );
				}
				//}
			} else {
				//attempt to update anonymous user to actual user!
				$args['user_action_message'] = $user_action_data['message'];
				do_action( 'uap_after_user_action_completed', $user_action_result, $user_id, $recipe_id, $trigger_log_id, $args );
			}
			//if $user_id found, update triggers, actions, recipe logs
			if ( $user_id && 0 !== $user_id ) {
				$args['user_id'] = $user_id;
				$table_name      = $wpdb->prefix . 'uap_trigger_log';
				$wpdb->update( $table_name, [ 'user_id' => $user_id ],
					[ 'ID' => $args['get_trigger_id'] ],
					[ '%d' ],
					[ '%d' ]
				);
				//attempt to update anonymous user to actual user!
				$table_name = $wpdb->prefix . 'uap_recipe_log';
				$wpdb->update( $table_name, [ 'user_id' => $user_id ],
					[ 'ID' => $args['recipe_log_id'] ],
					[ '%d' ],
					[ '%d' ]
				);
				//attempt to update anonymous user to actual user!
				$table_name = $wpdb->prefix . 'uap_trigger_log_meta';
				$wpdb->update( $table_name, [ 'user_id' => $user_id ],
					[ 'automator_trigger_log_id' => $args['get_trigger_id'] ],
					[ '%d' ],
					[ '%d' ]
				);

				$run_number  = $wpdb->get_var( "SELECT run_number FROM $table_name WHERE automator_trigger_log_id = " . $args['get_trigger_id'] . " AND user_id = $user_id" );
				$parsed_data = key_exists( 'parsed_data', $user_action_result ) ? $user_action_result['parsed_data'] : [];
				$wpdb->insert(
					$wpdb->prefix . 'uap_trigger_log_meta',
					array(
						'user_id'                  => $user_id,
						'automator_trigger_log_id' => $trigger_log_id,
						'automator_trigger_id'     => $trigger_id,
						'run_number'               => $run_number,
						'meta_key'                 => 'parsed_data',
						'meta_value'               => maybe_serialize( $parsed_data ),
					), array(
						'%d',
						'%d',
						'%d',
						'%d',
						'%s',
						'%s',
					)
				);
			}
		}

		// If all triggers for the recipe are completed
		if ( $maybe_continue_recipe_process && $this->triggers_completed( $recipe_id, $user_id, $recipe_log_id, $args ) ) {
			$this->complete_actions( $recipe_id, $user_id, $recipe_log_id, $args );
		}
	}

	/**
	 * @param $recipe_id
	 * @param $maybe_user_id
	 * @param $recipe_log_id
	 * @param $args
	 *
	 * @return array
	 */
	public function maybe_run_anonymous_recipe_user_actions( $recipe_id, $maybe_user_id, $recipe_log_id, $args ) {
		//we have a valid user_id, return user ID..
		//We expect a null or 0 user ID for anon function
		$p           = []; //for logs
		$m           = []; // for logs
		$fields      = get_post_meta( $recipe_id, 'fields', true );
		$user_action = get_post_meta( $recipe_id, 'source', true );
		$data        = [];
		$parsed_data = [];
		$fallback    = $fields['fallback'];
		if ( $fields ) {
			foreach ( $fields as $key => $field ) {
				if ( preg_match_all( "/\{\{\s*(.*?)\s*\}\}/", $field, $arr ) ) {
					$matches = $arr[1];
					$m[]     = $matches;
					foreach ( $matches as $match ) {
						$pieces = explode( ':', $match );
						$p[]    = $pieces;
						//Attempting to decode data!
						$val                   = apply_filters( 'automator_maybe_parse_token', '', $pieces, $recipe_id, $args, $maybe_user_id );
						$data[ $key ]          = $val;
						$parsed_data[ $field ] = $val;
					}
				} else {
					$data[ $key ]        = $field;
					$parsed_data[ $key ] = $field;
				}
			}
		}
		/*Utilities::log( [
			'$recipe_id'     => $recipe_id,
			'$maybe_user_id' => $maybe_user_id,
			'$fields'        => $fields,
			'$recipe_log_id' => $recipe_log_id,
			'$args'          => $args,
			'$data'          => $data,
			//'$user_data'     => $user_data,
			'$fallback'      => $fallback,
			'$user_action'   => $user_action,
			//$user_data,
		], '', true, 'complete_anon_user_actions' );*/

		$user_data = [];
		if ( ! empty( $data ) ) {

			$user_data['first_name']         = key_exists( 'firstName', $data ) ? $data['firstName'] : '';
			$user_data['last_name']          = key_exists( 'lastName', $data ) ? $data['lastName'] : '';
			$user_data['user_email']         = key_exists( 'email', $data ) ? $data['email'] : '';
			$user_data['user_login']         = key_exists( 'username', $data ) ? $data['username'] : '';
			$user_data['role']               = $fields['role'];
			$user_data['user_pass']          = key_exists( 'password', $data ) ? $data['password'] : apply_filters( 'uap_maybe_generate_anonymous_user_password', wp_generate_password(), $user_data );
			$user_data['prioritized_field']  = key_exists( 'prioritizedField', $fields ) ? $fields['prioritizedField'] : '';
			$user_data['unique_field_value'] = key_exists( 'uniqueFieldValue', $data ) ? $data['uniqueFieldValue'] : '';
			$user_data['unique_field']       = key_exists( 'uniqueField', $fields ) ? $fields['uniqueField'] : '';

			if ( 'newUser' === (string) $user_action ) {
				return $this->user_action_on_new_user( $user_data, $fallback, $parsed_data );
			} elseif ( 'existingUser' === (string) $user_action ) {
				return $this->user_action_on_existing_user( $user_data, $fallback, $parsed_data );
			}

		}

		//No user action ran.. return false
		return [ 'status' => false, 'data' => [] ];
	}

	/**
	 * @param $user_data
	 * @param $fallback
	 * @param array $parsed_data
	 *
	 * @return array
	 */
	public function user_action_on_new_user( $user_data, $fallback, $parsed_data = [] ) {
		$return  = [];
		$user_id = $this->verify_if_user_exists_for_new_user_action( $user_data );
		if ( ! $user_id ) {
			//@var $user_id not defined yet, insert new user
			$user_id = wp_insert_user( $user_data );

			if ( ! is_wp_error( $user_id ) ) {
				$data = [
					'user_id'  => $user_id,
					'fallback' => $fallback,
					'action'   => 'newUser',
					'message'  => sprintf( __( 'New user created: [%s]', 'uncanny-automator' ), $user_data['user_email'] ),
				];

				$return = [ 'status' => true, 'data' => $data, 'parsed_data' => $parsed_data ];
			} else {
				$data = [
					'user_id'  => email_exists( $user_data['user_email'] ),
					'fallback' => 'do-nothing',
					'action'   => 'newUser',
					'error'    => true,
					'message'  => sprintf( __( 'Create new user failed: [%s]. %s', 'uncanny-automator' ), $user_data['user_email'], $user_id->get_error_message() ),
				];

				$return = [ 'status' => false, 'data' => $data ];
			}
		} else {
			switch ( $fallback ) {
				case 'do-nothing':
					$user = get_user_by( 'ID', $user_id );
					$data = [
						'user_id'  => $user_id,
						'fallback' => $fallback,
						'action'   => 'newUser',
						'message'  => sprintf( __( 'Existing user found matching: [%s], do nothing', 'uncanny-automator' ), $user->user_login ),
					];

					$return = [ 'status' => false, 'data' => $data ];
					break;
				case 'select-existing-user':
					//Select Existing User
					$status = true;
					$user   = get_user_by( 'ID', $user_id );
					$data   = [
						'user_id'  => $user_id,
						'fallback' => $fallback,
						'action'   => 'newUser',
						'message'  => sprintf( __( 'Existing user found matching [%s], selected existing user', 'uncanny-automator' ), $user->user_email ),
					];

					if ( in_array( 'administrator', $user->roles ) ) {
						$data['message']  = sprintf( __( 'Existing user found matching [%s], cannot select Administrators', 'uncanny-automator' ), $user->user_email );
						$data['fallback'] = 'do-nothing';
						$status           = false;
					}

					$return = [ 'status' => $status, 'data' => $data, 'parsed_data' => $parsed_data ];
					break;
			}
		}

		return $return;
	}

	/**
	 * @param $user_data
	 * @param $fallback
	 * @param array $parsed_data
	 *
	 * @return array
	 */
	public function user_action_on_existing_user( $user_data, $fallback, $parsed_data = [] ) {
		$user_id = $this->verify_if_user_exists_for_existing_user_action( $user_data );
		$return  = [];
		if ( ! $user_id ) {
			//User not found
			switch ( $fallback ) {
				case 'do-nothing':
					$data = [
						'user_id'  => 0,
						'fallback' => $fallback,
						'action'   => 'existingUser',
						'message'  => sprintf( __( 'User not found matching [%s], do nothing', 'uncanny-automator' ), $user_data['unique_field_value'] ),
					];

					$return = [ 'status' => false, 'data' => $data ];
					break;
				case 'create-new-user':
					//Create a new user
					$user_id = wp_insert_user( $user_data );
					if ( ! is_wp_error( $user_id ) ) {
						$data = [
							'user_id'  => $user_id,
							'fallback' => $fallback,
							'action'   => 'existingUser',
							'message'  => sprintf( __( 'User not found matching [%s], new user created', 'uncanny-automator' ), $user_data['user_email'] ),
						];

						$return = [ 'status' => true, 'data' => $data, 'parsed_data' => $parsed_data ];
					} else {
						$data = [
							'user_id'  => 0,
							'fallback' => $fallback,
							'action'   => 'existingUser',
							'error'    => true,
							'message'  => sprintf( __( 'Create new user failed: [%s]. %s', 'uncanny-automator' ), $user_data['user_email'], $user_id->get_error_message() ),
						];

						$return = [ 'status' => false, 'data' => $data ];
					}
					break;
			}
		} else {
			//User found!
			$status = true;
			$user   = get_user_by( 'ID', $user_id );
			$data   = [
				'user_id'  => $user_id,
				'fallback' => $fallback,
				'action'   => 'existingUser',
				'message'  => sprintf( __( 'User found matching [%s], selecting user', 'uncanny-automator' ), $user->user_email ),
			];

			if ( in_array( 'administrator', $user->roles ) ) {
				$data['message']  = sprintf( __( 'Existing user found matching [%s], cannot select Administrators', 'uncanny-automator' ), $user->user_email );
				$data['fallback'] = 'do-nothing';
				$status           = false;
			}

			$return = [ 'status' => $status, 'data' => $data, 'parsed_data' => $parsed_data ];
		}

		return $return;
	}

	/**
	 * @param $user_data
	 * @param $user_action
	 *
	 * @return bool|false|int
	 */
	public function verify_if_user_exists_for_new_user_action( $user_data ) {
		//If there's a priority to match
		$priority = 'email';
		if ( key_exists( 'prioritized_field', $user_data ) ) {
			$priority = (string) $user_data['prioritized_field'];
		}

		switch ( $priority ) {
			case 'username':
				$user_id = username_exists( $user_data['user_login'] );
				if ( $user_id ) {
					return $user_id;
				} elseif ( ! empty( $user_data['user_email'] ) ) {
					$user_id = email_exists( $user_data['user_email'] );
					if ( $user_id ) {
						return $user_id;
					}
				}

				break;

			case 'email':
			default:
				$user_id = email_exists( $user_data['user_email'] );
				if ( $user_id ) {
					return $user_id;
				} elseif ( empty( $user_data['user_login'] ) ) {
					$user_id = username_exists( $user_data['user_login'] );
					if ( $user_id ) {
						return $user_id;
					}
				}

				break;
		}

		return false;

	}

	/**
	 * @param $user_data
	 * @param $user_action
	 *
	 * @return bool|false|int
	 */
	public function verify_if_user_exists_for_existing_user_action( $user_data ) {
		//If there's a priority to match
		$unique_field = ( key_exists( 'unique_field', $user_data ) && ! empty( $user_data['unique_field'] ) ) ? $user_data['unique_field'] : 'email';
		$value        = $user_data['unique_field_value'];

		if ( empty( $value ) ) {
			return false;
		}

		switch ( $unique_field ) {
			case 'username':
				$user_id = username_exists( $value );

				return ! is_wp_error( $user_id ) ? $user_id : false;
				break;
			case 'id':
				$user_id = get_user_by( 'ID', $value );

				return ! is_wp_error( $user_id ) ? $user_id : false;

				break;
			case 'email':
			default:
				$user_id = email_exists( $value );

				return ! is_wp_error( $user_id ) ? $user_id : false;
				break;
		}
	}

	/**
	 * Complete the action for the user
	 *
	 * @param $user_id       null||int
	 * @param $action_data   null||int
	 * @param $recipe_id     null||int
	 * @param $error_message string
	 * @param $recipe_log_id null
	 *
	 * @param array $args
	 *
	 * @return null
	 */
	public function complete_action( $user_id = null, $action_data = null, $recipe_id = null, $error_message = '', $recipe_log_id = null, $args = [] ) {

		// Set user ID
		if ( is_null( $user_id ) ) {
			$user_id = get_current_user_id();
		}
		/*Utilities::log( [
			'$user_id'       => $user_id,
			'$action_data'   => $action_data,
			'$recipe_id'     => $recipe_id,
			'$recipe_log_id' => $recipe_log_id,
			'$error_message' => $error_message,
			'$args'          => $args,
		], '', true, 'step3-a' );*/

		$action_id = (int) $action_data['ID'];

		if ( null === $action_id || ! is_numeric( $action_id ) ) {
			Utilities::log( 'ERROR: You are trying to complete a trigge without providing a trigger_id', 'complete_uap_action ERROR', false, 'uap-errors' );

			return null;
		}

		if ( null === $recipe_id || ! is_numeric( $recipe_id ) ) {
			Utilities::log( 'ERROR: You are trying to complete a trigge without providing a recipe_id', 'complete_uap_action ERROR', false, 'uap-errors' );

			return null;
		}

		if ( is_null( $recipe_log_id ) && key_exists( 'recipe_log_id', $action_data ) ) {
			$recipe_log_id = $action_data['recipe_log_id'];
		}

		if ( is_null( $recipe_log_id ) || empty( $recipe_log_id ) ) {
			return null;
		}

		if ( empty( $args ) && key_exists( 'args', $action_data ) ) {
			$args = $action_data['args'];
		}

		// The trigger is about to be completed
		do_action( 'uap_before_action_completed', $user_id, $action_id, $recipe_id, $error_message, $recipe_log_id, $args );

		$completed = 0;

		if ( ( is_array( $action_data ) && key_exists( 'do-nothing', $action_data ) ) ) {
			if ( key_exists( 'complete_with_errors', $action_data ) ) {
				$completed = 2;
			} else {
				$completed = 9;
			}

			$error_message = key_exists( 'user_action_message', $args ) ? $args['user_action_message'] : $error_message;
		} elseif ( empty( $error_message ) ) {
			$completed = 1;
			if ( key_exists( 'user_action_message', $args ) && ! empty( $args['user_action_message'] ) ) {
				$error_message = $args['user_action_message'];
			}
		}

		global $wpdb;

		$table_name = $wpdb->prefix . 'uap_action_log';
		/*Utilities::log( [
			'ala'   => array(
				'date_time'               => date( 'Y-m-d H:i:s', current_time( 'timestamp' ) ),
				'user_id'                 => $user_id,
				'automator_action_id'     => $action_id,
				'automator_recipe_id'     => $recipe_id,
				'automator_recipe_log_id' => $recipe_log_id,
				'completed'               => $completed,
				'error_message'           => $error_message
			),
			'$args' => $args
		], '', true, 'step3-a' );*/

		$inserted = $wpdb->insert( $table_name,
			array(
				'date_time'               => date( 'Y-m-d H:i:s', current_time( 'timestamp' ) ),
				'user_id'                 => $user_id,
				'automator_action_id'     => $action_id,
				'automator_recipe_id'     => $recipe_id,
				'automator_recipe_log_id' => $recipe_log_id,
				'completed'               => $completed,
				'error_message'           => ! empty( $error_message ) ? $error_message : ''
			), array(
				'%s',
				'%d',
				'%d',
				'%d',
				'%d',
				'%d',
				'%s',
			) );

		// The actions is now completed
		do_action( 'uap_action_completed', $user_id, $action_id, $recipe_id, $error_message, $args );

	}

	/**
	 * Are all triggers in the recipe completed
	 *
	 * @param int $recipe_id null||int
	 * @param int $user_id null||int
	 * @param int $recipe_log_id null||int
	 *
	 * @param array $args
	 *
	 * @return bool|null
	 */
	public function triggers_completed( $recipe_id = 0, $user_id = 0, $recipe_log_id = 0, $args = [] ) {

		if ( null === $recipe_id || ! is_numeric( $recipe_id ) ) {
			Utilities::log( 'ERROR: You are trying to check if triggers are completed without providing a recipe_id', 'triggers_completed ERROR', false, 'uap-errors' );

			return null;
		}

		// Set user ID
		if ( is_null( $user_id ) ) {
			$user_id = get_current_user_id();
		}

		// No user id is aviable.
		/*if ( 0 === $user_id ) {
			Utilities::log( 'ERROR: You are trying to check if triggers are completed when a there is no logged in user.', 'triggers_completed ERROR', false, 'uap-errors' );

			return null;
		}*/

		$recipe_triggers = $this->get_recipe_data( 'uo-trigger', $recipe_id );

		// By default the recipe will complete unless there is a trigger that is live(publish status) and its NOT completed
		$triggers_completed = true;

		foreach ( $recipe_triggers as $recipe_trigger ) {
			if ( 'publish' === (string) $recipe_trigger['post_status'] ) {
				$trigger_integration = $recipe_trigger['meta']['integration'];

				if ( 0 === $this->plugin_status->get( $trigger_integration ) ) {
					// The plugin for this trigger is NOT active
					Utilities::log( 'ERROR: You are trying to complete ' . $recipe_trigger['meta']['code'] . ' and the plugin ' . $trigger_integration . ' is not active. @recipe_id ' . $recipe_id, 'complete_trigger ERROR', false, 'uap-errors' );
				}

				$trigger_completed = $this->is_trigger_completed( $user_id, $recipe_trigger['ID'], $recipe_id, $recipe_log_id, $args );
				if ( ! $trigger_completed ) {
					return false;
				}
			}
		}

		return $triggers_completed;

	}

	/**
	 * Get the trigger for the user
	 *
	 * @param $user_id       null||int
	 * @param $trigger_id    null||int
	 * @param $meta_key      null||string
	 * @param $recipe_log_id null||int
	 *
	 * @return null|int
	 */
	public function trigger_meta_id( $user_id = null, $trigger_id = null, $meta_key = null, $recipe_log_id = null ) {

		// Set user ID
		if ( null === $user_id ) {
			$user_id = get_current_user_id();
		}

		// No user id is aviable.
		if ( 0 === $user_id ) {
			Utilities::log( 'ERROR: You are trying to get trigger meta ID when a there is no logged in user.', 'get_trigger_meta_id ERROR', false, 'uap-errors' );

			return null;
		}

		if ( null === $trigger_id || ! is_numeric( $trigger_id ) ) {
			Utilities::log( 'ERROR: You are trying to get trigger meta ID without providing a trigger_id', 'get_trigger_meta_id ERROR', false, 'uap-errors' );

			return null;
		}

		if ( null === $meta_key || ! is_string( $meta_key ) ) {
			Utilities::log( 'ERROR: You are trying to get trigger meta ID without providing a meta_key', 'get_trigger_meta_id ERROR', false, 'uap-errors' );

			return null;
		}

		global $wpdb;
		$table_name = $wpdb->prefix . 'uap_trigger_log_meta';
		$results    = $wpdb->get_var( $wpdb->prepare( "SELECT ID FROM $table_name WHERE user_id = %d AND meta_key LIKE %s AND automator_trigger_id = %d", $user_id, $meta_key, $trigger_id ) );

		if ( null !== $results ) {
			return (int) $results;
		}

		return $results;
	}

	/**
	 * Insert the trigger for the user
	 *
	 * @param $args
	 *
	 * @return null||int
	 */
	public function insert_trigger_meta( $args ) {
		$user_id        = $args['user_id'];
		$trigger_id     = $args['trigger_id'];
		$meta_key       = $args['meta_key'];
		$meta_value     = $args['meta_value'];
		$trigger_log_id = $args['trigger_log_id'];
		$run_number     = $args['run_number'];
		// Set user ID
		if ( is_null( $user_id ) ) {
			$user_id = get_current_user_id();
		}

		// No user id is aviable.
		/*if ( 0 === $user_id ) {
			Utilities::log( 'ERROR: You are trying to insert trigger meta when a there is no logged in user.', 'insert_trigger_meta ERROR', false, 'uap-errors' );

			return null;
		}*/

		if ( null === $trigger_id || ! is_numeric( $trigger_id ) ) {
			Utilities::log( 'ERROR: You are trying to insert trigger meta without providing a trigger_id', 'insert_trigger_meta ERROR', false, 'uap-errors' );

			return null;
		}

		if ( null === $meta_key || ! is_string( $meta_key ) ) {
			Utilities::log( 'ERROR: You are trying to insert trigger meta without providing a meta_key', 'insert_trigger_meta ERROR', false, 'uap-errors' );

			return null;
		}

		if ( null === $meta_value ) {
			Utilities::log( 'ERROR: You are trying to insert trigger meta without providing a meta_value', 'insert_trigger_meta ERROR', false, 'uap-errors' );

			return null;
		}

		if ( is_numeric( $trigger_log_id ) ) {
			global $wpdb;
			$table_name = $wpdb->prefix . 'uap_trigger_log_meta';
			$inserted   = $wpdb->insert( $table_name,
				array(
					'user_id'                  => $user_id,
					'automator_trigger_log_id' => $trigger_log_id,
					'automator_trigger_id'     => $trigger_id,
					'run_number'               => $run_number,
					'meta_key'                 => $meta_key,
					'meta_value'               => $meta_value,
				), array(
					'%d',
					'%d',
					'%d',
					'%d',
					'%s',
					'%s',
				) );

			if ( false !== $inserted ) {
				return $wpdb->insert_id;
			}
		}

		return null;


	}

	/**
	 * Update the trigger for the user
	 *
	 * @param $user_id       null||int
	 * @param $trigger_id    null||int
	 * @param $meta_key      null||string
	 * @param $meta_value    string
	 * @param $trigger_log_id
	 *
	 * @return null
	 */
	public function update_trigger_meta( $user_id = null, $trigger_id = null, $meta_key = null, $meta_value = '', $trigger_log_id = null ) {

		// Set user ID
		if ( null === $user_id ) {
			$user_id = get_current_user_id();
		}

		// No user id is aviable.
		if ( 0 === $user_id ) {
			Utilities::log( 'ERROR: You are trying to update trigger meta when a there is no logged in user.', 'update_trigger_meta ERROR', false, 'uap-errors' );

			return null;
		}

		if ( null === $trigger_id || ! is_numeric( $trigger_id ) ) {
			Utilities::log( 'ERROR: You are trying to update trigger meta without providing a trigger_id', 'update_trigger_meta ERROR', false, 'uap-errors' );

			return null;
		}

		if ( null === $meta_key || ! is_string( $meta_key ) ) {
			Utilities::log( 'ERROR: You are trying to update trigger meta without providing a meta_key', 'update_trigger_meta ERROR', false, 'uap-errors' );

			return null;
		}

		global $wpdb;
		$update = array(
			'meta_value' => $meta_value,
		);

		$where = array(
			'user_id'              => $user_id,
			'automator_trigger_id' => $trigger_id,
			'meta_key'             => $meta_key,
		);

		if ( ! empty( $trigger_log_id ) && is_numeric( $trigger_log_id ) ) {
			$where['automator_trigger_log_id'] = (int) $trigger_log_id;
		}

		$update_format = array(
			'%d',
		);

		$where_format = array(
			'%d',
			'%d',
			'%s',
		);

		if ( ! empty( $trigger_log_id ) && is_numeric( $trigger_log_id ) ) {
			$where_format[] = '%d';
		}
		$table_name = $wpdb->prefix . 'uap_trigger_log_meta';
		$wpdb->update( $table_name,
			$update,
			$where,
			$update_format,
			$where_format
		);
	}

	/**
	 *
	 * Complete a trigger once all validation & trigger entry added
	 * and number of times met, complete the trigger
	 *
	 * @param $args
	 *
	 * @return bool
	 */
	public function maybe_trigger_complete( $args ) {
		$is_signed_in = key_exists( 'is_signed_in', $args ) ? $args['is_signed_in'] : false;

		if ( empty( $args ) && false === $is_signed_in ) {
			return false;
		}

		$this->complete_trigger( $args );
	}

	/**
	 *
	 * Matches recipes against trigger meta/code. If a recipe is found and not completed,
	 * add a trigger entry in to the DB and matches number of times.
	 *
	 * @param      $args
	 * @param bool $mark_trigger_complete
	 *
	 * @return array|bool|int|null
	 */
	public function maybe_add_trigger_entry( $args, $mark_trigger_complete = true ) {
		$is_signed_in = key_exists( 'is_signed_in', $args ) ? true : false;
		$is_anonymous = false;
		if ( ! is_user_logged_in() && false === $is_signed_in ) {
			$args['user_id']           = 0;
			$args['mark_as_anonymous'] = true;
			$is_anonymous              = true;
		}

		$check_trigger_code = key_exists( 'code', $args ) ? $args['code'] : null;
		$trigger_meta       = key_exists( 'meta', $args ) ? $args['meta'] : null;
		$post_id            = key_exists( 'post_id', $args ) ? $args['post_id'] : 0;
		$user_id            = key_exists( 'user_id', $args ) ? $args['user_id'] : wp_get_current_user()->ID;
		$matched_recipe_id  = key_exists( 'recipe_to_match', $args ) ? (int) $args['recipe_to_match'] : null;
		$matched_trigger_id = key_exists( 'trigger_to_match', $args ) ? (int) $args['trigger_to_match'] : null;
		$ignore_post_id     = key_exists( 'ignore_post_id', $args ) ? true : false;
		$get_trigger_id     = null;
		$result             = [];

		if ( is_null( $check_trigger_code ) ) {
			return null;
		}

		$args = [
			'code'             => $check_trigger_code,
			'meta'             => $trigger_meta,
			'post_id'          => $post_id,
			'user_id'          => $user_id,
			'recipe_to_match'  => $matched_recipe_id,
			'trigger_to_match' => $matched_trigger_id,
			'ignore_post_id'   => $ignore_post_id,
			'is_signed_in'     => $is_signed_in,
		];

		//$recipes = $this->get_recipes_data( $force_new_data, $matched_recipe_id, $check_trigger_code );
		$recipes = $this->get->recipes_from_trigger_code( $check_trigger_code );
		/*Utilities::log( [
			'$recipes'      => $recipes,
			'$args'         => $args,
			'$is_signed_in' => $is_signed_in,
			'$is_anonymous' => $is_anonymous,
			'$is_anonymous' => $is_anonymous,
		], '$recipes', true, 'recipes1' );*/
		foreach ( $recipes as $recipe ) {
			//loop only published
			if ( 'publish' !== $recipe['post_status'] ) {
				continue;
			}

			/*if ( 'anonymous' === (string) $recipe['recipe_type'] && ! $is_anonymous ) {
				//If it's an anonymous recipe & user is logged in.. skip recipe
				continue;
			} else*/
			if ( 'user' === (string) $recipe['recipe_type'] && $is_anonymous ) {
				//If it's user recipe & user is not logged in.. skip recipe
				continue;
			}

			$recipe_id           = $recipe['ID'];
			$maybe_recipe_log    = $this->maybe_create_recipe_log_entry( $recipe_id, $user_id, true, $args, true );
			$maybe_recipe_log_id = (int) $maybe_recipe_log['recipe_log_id'];
			/*Utilities::log( [
				//'$maybe_recipe_log_id' => $maybe_recipe_log_id,
				'$recipe_id'           => $recipe_id,
				'$recipe'              => $recipe,
				'$maybe_recipe_log'    => $maybe_recipe_log,
				'$maybe_recipe_log_id' => $maybe_recipe_log_id,
			], '--BEFORE11--', true, 'step--1' );*/
			foreach ( $recipe['triggers'] as $trigger ) {
				if ( ! empty( $matched_trigger_id ) && is_numeric( $matched_trigger_id ) && (int) $matched_trigger_id !== (int) $trigger['ID'] ) {
					continue;
				}

				$trigger_id          = $trigger['ID'];
				$trigger_post_status = $trigger['post_status'];

				if ( 'publish' !== $trigger_post_status ) {
					continue;
				}
				/*Utilities::log( [
					//'$maybe_recipe_log_id' => $maybe_recipe_log_id,
					'$recipe_id'        => $recipe_id,
					'$trigger'          => $trigger,
					'$maybe_recipe_log' => $maybe_recipe_log,
				], '--BEFORE11--', true, 'step0' );*/
				//Insert a recipe log, needed for multiple run recipes
				$get_trigger_id = $this->get_trigger_id( $args, $trigger, $recipe_id, $maybe_recipe_log_id, $ignore_post_id, $is_anonymous );
				/*Utilities::log( [
					//'$maybe_recipe_log_id' => $maybe_recipe_log_id,
					'$recipe_id'      => $recipe_id,
					'$trigger'        => $trigger,
					'$get_trigger_id' => $get_trigger_id,
				], '--BEFORE--', true, 'step0' );*/


				if ( is_array( $get_trigger_id ) && false === $get_trigger_id['result'] ) {
					$result[] = $get_trigger_id;
					continue;
				}

				if ( ! $maybe_recipe_log['existing'] ) {
					//trigger validated.. add recipe log ID now!
					//$recipe_log_id = $this->maybe_create_recipe_log_entry( $recipe_id, $user_id, true, $args, true, $maybe_recipe_log_id );
					$recipe_log_details = $this->maybe_create_recipe_log_entry( $recipe_id, $user_id, true, $args );
					$recipe_log_id      = (int) $recipe_log_details['recipe_log_id'];
					//running again--after $recipe_log_id
					$get_trigger_id = $this->get_trigger_id( $args, $trigger, $recipe_id, $maybe_recipe_log_id, $ignore_post_id, $is_anonymous );
				} else {
					$recipe_log_id = $maybe_recipe_log_id;
				}

				/*Utilities::log( [
					'$maybe_recipe_log_id' => $maybe_recipe_log_id,
					'$recipe_log_id'       => $recipe_log_id,
					'$get_trigger_id'      => $get_trigger_id,
					'$result'              => $result,
				], '', true, 'step0' );*/

				$get_trigger_id = $get_trigger_id['get_trigger_id'];

				$numtimes_arg = [
					'recipe_id'      => $recipe_id,
					'trigger_id'     => $trigger_id,
					'trigger'        => $trigger,
					'user_id'        => $user_id,
					'recipe_log_id'  => $recipe_log_id,
					'trigger_log_id' => $get_trigger_id,
					'is_signed_in'   => $is_signed_in,
				];

				$trigger_steps_completed = $this->maybe_trigger_num_times_completed( $numtimes_arg, $is_anonymous );
				//If -1 / Any option is used, save it's entry for tokens
				if ( ( isset( $trigger['meta'][ $trigger_meta ] ) && intval( '-1' ) === intval( $trigger['meta'][ $trigger_meta ] ) ) && true === $trigger_steps_completed['result'] ) {
					$meta_arg = [
						'recipe_id'      => $recipe_id,
						'trigger_id'     => $trigger_id,
						'user_id'        => $user_id,
						'recipe_log_id'  => $recipe_log_id,
						'trigger_log_id' => $get_trigger_id,
						'post_id'        => $post_id,
						'trigger'        => $trigger,
						'is_signed_in'   => $is_signed_in,
						'meta'           => $trigger_meta,
						'run_number'     => $this->get->next_run_number( $recipe_id, $user_id, true ),
					];
					#Utilities::log( $meta_arg, '$meta_arg', true, 'token-fixes' );

					$meta_results = $this->maybe_trigger_add_any_option_meta( $meta_arg, $trigger_meta );
					if ( false === $meta_results['result'] ) {
						Utilities::log( 'ERROR: You are trying to add entry ' . $trigger['meta'][ $trigger_meta ] . ' and post_id = ' . $post_id . '.', 'uap_maybe_add_meta_entry ERROR', false, 'uap-errors' );
					}

				}

				do_action( 'uap_after_trigger_run', $check_trigger_code, $post_id, $user_id, $trigger_meta );

				if ( true === $trigger_steps_completed['result'] ) {
					$args['get_trigger_id'] = $get_trigger_id;
					$args['recipe_id']      = $recipe_id;
					$args['trigger_id']     = $trigger_id;
					$args['recipe_log_id']  = $recipe_log_id;
					$args['post_id']        = $post_id;
					$args['is_signed_in']   = $is_signed_in;
					$args['run_number']     = $this->get->next_run_number( $recipe_id, $user_id, true );

					if ( 1 === + $mark_trigger_complete ) {
						$this->maybe_trigger_complete( $args );
					} else {
						$result[] = [ 'result' => true, 'args' => $args ];
					}
				}
			}
		}

		return $result;
	}

	/**
	 * @param $args
	 * @param $trigger
	 * @param $recipe_id
	 * @param $maybe_recipe_log_id
	 * @param $ignore_post_id
	 * @param $is_anonymous
	 *
	 * @return array
	 */
	public function get_trigger_id( $args, $trigger, $recipe_id, $maybe_recipe_log_id, $ignore_post_id, $is_anonymous ) {

		if ( $ignore_post_id ) {
			$get_trigger_id = $this->maybe_validate_trigger_without_postid( $args, $trigger, $recipe_id, $maybe_recipe_log_id, $is_anonymous );
		} else {
			$get_trigger_id = $this->maybe_validate_trigger( $args, $trigger, $recipe_id, $maybe_recipe_log_id, $is_anonymous );
		}

		return $get_trigger_id;
	}

	/**
	 * @param int $recipe_id
	 * @param int $user_id
	 * @param bool $create_recipe
	 * @param array $args
	 * @param bool $maybe_simulate
	 * @param null $maybe_add_log_id
	 *
	 * @return array
	 * @author Saad S. on Nov 15th, 2019
	 *
	 * Added $maybe_simulate in order to avoid unnecessary recipe logs in database.
	 * It'll return existing $recipe_log_id if there's one for a user & recipe, or
	 * simulate an ID for the next run.. The reason for simulate is to avoid unnecessary
	 * recipe_logs in the database since we insert recipe log first & check if trigger
	 * is valid after which means, recipe log is added and not used in this run.
	 * Once trigger is validated.. I pass $maybe_simulate ID to $maybe_add_log_id
	 * and insert recipe log at this point.
	 *
	 * @since 2.0
	 */
	public function maybe_create_recipe_log_entry( $recipe_id, $user_id, $create_recipe = true, $args = [], $maybe_simulate = false, $maybe_add_log_id = null ) {
		global $wpdb;
		$anon = false;
		if ( key_exists( 'mark_as_anonymous', $args ) ) {
			$anon = true;
		}

		$recipe_log_id = $wpdb->get_var( $wpdb->prepare( "SELECT ID FROM {$wpdb->prefix}uap_recipe_log WHERE completed != %d  AND automator_recipe_id = %d AND user_id = %d", 1, $recipe_id, $user_id ) );
		if ( ! $anon && $recipe_log_id && 0 !== absint( $user_id ) ) {
			return [ 'existing' => true, 'recipe_log_id' => $recipe_log_id, ];
		} elseif ( true === $maybe_simulate ) {
			/*
			 * @since 2.0
			 * @author Saad S.
			 */
			if ( ! is_null( $maybe_add_log_id ) ) {
				return [
					'existing'      => false,
					'recipe_log_id' => $this->insert_recipe_log( $recipe_id, $user_id, $maybe_add_log_id ),
				];
				//return $this->insert_recipe_log( $recipe_id, $user_id, $maybe_add_log_id );
			} else {
				$recipe_log_id = $wpdb->get_var( "SELECT `AUTO_INCREMENT` FROM  INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = '" . DB_NAME . "' AND TABLE_NAME   = '{$wpdb->prefix}uap_recipe_log' LIMIT 0,1;" );
				//Utilities::log( $recipe_log_id, '$recipe_log_id', true, 'recipe_log_id' );
				if ( $recipe_log_id ) {
					return [ 'existing' => false, 'recipe_log_id' => $recipe_log_id, ];
					//return $recipe_log_id;
				}
			}
		} elseif ( true === $create_recipe ) {
			return [
				'existing'      => false,
				'recipe_log_id' => $this->insert_recipe_log( $recipe_id, $user_id, null, $anon ),
			];
			//return $this->insert_recipe_log( $recipe_id, $user_id, null, $anon );
		}

		return [ 'existing' => false, 'recipe_log_id' => null ];
		//return null;
	}

	/**
	 * @param $recipe_id
	 * @param $user_id
	 * @param null $maybe_add_log_id
	 * @param bool $maybe_anonymous
	 *
	 * @return int
	 */
	public function insert_recipe_log( $recipe_id, $user_id, $maybe_add_log_id = null, $maybe_anonymous = false ) {
		global $wpdb;
		$table_name = $wpdb->prefix . 'uap_recipe_log';

		if ( is_null( $maybe_add_log_id ) && true === $maybe_anonymous ) {
			$num_times_recipe_run = false;
		} else {
			$results = $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(completed) FROM $table_name WHERE user_id = %d AND automator_recipe_id = %d", $user_id, $recipe_id ) );
			if ( 0 !== absint( $user_id ) ) {
				$num_times_recipe_run = $this->utilities->recipe_number_times_completed( $recipe_id, $results );
			} else {
				$num_times_recipe_run = false;
			}
		}

		if ( ! $num_times_recipe_run ) {
			$run_number = $this->get->next_run_number( $recipe_id, $user_id );

			$insert = array(
				'date_time'           => '0000-00-00 00:00:00',
				'user_id'             => $user_id,
				'automator_recipe_id' => $recipe_id,
				'completed'           => - 1,
				'run_number'          => $run_number,
			);

			$format = array(
				'%s',
				'%d',
				'%d',
				'%d',
			);

			if ( ! is_null( $maybe_add_log_id ) ) {
				$insert['ID'] = $maybe_add_log_id;
				$format[]     = '%d';
			}

			$r = $wpdb->insert( $table_name, $insert, $format );


			return (int) $wpdb->insert_id;
		}

		return null;
	}

	/**
	 * @param      $recipe_id
	 * @param      $user_id
	 * @param      $recipe_log_id
	 * @param bool $change_to_zero
	 */
	public function maybe_change_recipe_log_to_zero( $recipe_id, $user_id, $recipe_log_id, $change_to_zero = false ) {
		global $wpdb;
		$if_exists = $wpdb->get_var( $wpdb->prepare( "SELECT ID FROM {$wpdb->prefix}uap_recipe_log WHERE completed = %d AND automator_recipe_id = %d AND user_id = %d", - 1, $recipe_id, $user_id ) );
		if ( ! empty( $if_exists ) && (int) $if_exists === (int) $recipe_log_id && true === $change_to_zero ) {

			$result = $wpdb->update(
				$wpdb->prefix . 'uap_recipe_log',
				array(
					'completed' => 0,
				),
				array(
					'ID'                  => $recipe_log_id,
					'automator_recipe_id' => $recipe_id,
					//'user_id'             => $user_id,
				),
				array(
					'%d',
				),
				array(
					'%d',
					'%d',
					//'%d',
				)
			);
		}
	}

	/**
	 *
	 * Validate (int) values trigger v/s (int) trigger['meta'].
	 * If matched add value to trigger log table
	 *
	 * @param array $args
	 * @param null $trigger
	 * @param null $recipe_id
	 * @param null $recipe_log_id
	 * @param bool $is_anonymous
	 *
	 * @return array
	 */
	public function maybe_validate_trigger( $args = [], $trigger = null, $recipe_id = null, $recipe_log_id = null, $is_anonymous = false ) {

		if ( $is_anonymous ) {
			$args['user_id']           = 0;
			$args['mark_as_anonymous'] = true;
		}

		if ( empty( $args ) || null === $trigger || null === $recipe_id ) {
			return [
				'result' => false,
				'error'  => __( 'One of the required field is missing.', 'uncanny-automator' ),
			];
		}

		$check_trigger_code  = $args['code'];
		$trigger_meta        = $args['meta'];
		$post_id             = $args['post_id'];
		$user_id             = $args['user_id'];
		$trigger_id          = $trigger['ID'];
		$trigger_code        = $trigger['meta']['code'];
		$trigger_integration = $trigger['meta']['integration'];

		// Skip completion if the plugin is not active
		if ( 0 === $this->plugin_status->get( $trigger_integration ) ) {
			// The plugin for this trigger is NOT active
			Utilities::log( 'ERROR: You are trying to complete ' . $trigger['meta']['code'] . ' and the plugin ' . $trigger_integration . ' is not active. ', 'uap_do_trigger_log ERROR', false, 'uap-errors' );

			return [
				'result' => false,
				'error'  => __( 'Plugin is not active.', 'uncanny-automator' ),
			];
		}

		// Stop here if the trigger was already completed
		$is_trigger_completed = $this->is_trigger_completed( $user_id, $trigger_id, $recipe_id, $recipe_log_id, $args );
		if ( $is_trigger_completed ) {
			return [
				'result' => false,
				'error'  => __( 'Trigger is completed.', 'uncanny-automator' ),
			];
		}

		// Skip if the executed trigger doesn't match
		if ( $check_trigger_code !== $trigger_code ) {
			return [
				'result' => false,
				'error'  => __( 'Trigger isn\'t matched.', 'uncanny-automator' ),
			];
		}

		// The post ID the current user needs to visit
		if ( key_exists( $trigger_meta, $trigger['meta'] ) ) {
			$trigger_post_id = intval( $trigger['meta'][ $trigger_meta ] );
		} else {
			$trigger_post_id = 0;
		}

		if ( intval( '-1' ) !== intval( $trigger_post_id ) ) {
			if ( $trigger_post_id !== $post_id ) {
				return [
					'result' => false,
					'error'  => __( 'Trigger not matched.', 'uncanny-automator' ),
				];
			}
		}

		return $this->maybe_get_trigger_id( $user_id, $trigger_id, $recipe_id, $recipe_log_id );
	}

	/**
	 *
	 * Validate recipe post ID when ignore post id is passed.
	 * This is mostly going to be used when user/dev done validation in trigger
	 * and passes recipe IDs for this to be validated and added to trigger log DB.
	 *
	 * @param array $args
	 * @param null $trigger
	 * @param null $recipe_id
	 * @param null $recipe_log_id
	 * @param bool $is_anonymous
	 *
	 * @return array
	 */
	public function maybe_validate_trigger_without_postid( $args = [], $trigger = null, $recipe_id = null, $recipe_log_id = null, $is_anonymous = false ) {

		if ( $is_anonymous ) {
			$args['user_id']           = 0;
			$args['mark_as_anonymous'] = true;
			/*return [
				'result' => false,
				'error'  => __( 'User is not logged in.', 'uncanny-automator' ),
			];*/
		}

		if ( empty( $args ) || null === $trigger || null === $recipe_id ) {
			return [
				'result' => false,
				'error'  => __( 'One of the required field is missing.', 'uncanny-automator' ),
			];
		}

		$check_trigger_code  = $args['code'];
		$trigger_meta        = $args['meta'];
		$user_id             = $args['user_id'];
		$matched_recipe_id   = $args['recipe_to_match'];
		$matched_trigger_id  = $args['trigger_to_match'];
		$trigger_id          = is_numeric( $matched_trigger_id ) ? (int) $matched_trigger_id : $trigger['ID'];
		$trigger_code        = $trigger['meta']['code'];
		$trigger_integration = $trigger['meta']['integration'];

		// Skip completion if the plugin is not active
		if ( 0 === $this->plugin_status->get( $trigger_integration ) ) {
			// The plugin for this trigger is NOT active
			Utilities::log( 'ERROR: You are trying to complete ' . $trigger['meta']['code'] . ' and the plugin ' . $trigger_integration . ' is not active. ', 'uap_do_trigger_log ERROR', false, 'uap-errors' );

			return [
				'result' => false,
				'error'  => __( 'Plugin is not active.', 'uncanny-automator' ),
			];
		}

		/*if ( is_null( $recipe_log_id ) || ! is_numeric( $recipe_log_id ) ) {
			$recipe_log_id = $this->maybe_create_recipe_log_entry( $recipe_id, $user_id, true );
		}*/

		// Stop here if the trigger was already completed
		$is_trigger_completed = $this->is_trigger_completed( $user_id, $trigger_id, $recipe_id, $recipe_log_id, $args );

		if ( $is_trigger_completed ) {
			return [
				'result' => false,
				'error'  => __( 'Trigger is completed.', 'uncanny-automator' ),
			];
		}
		// Skip if the executed trigger doesn't match
		if ( (string) $check_trigger_code !== (string) $trigger_code ) {
			return [
				'result' => false,
				'error'  => sprintf( __( '%s AND %s triggers not matched.', 'uncanny-automator' ), $check_trigger_code, $trigger_code ),
			];
		}

		if ( 0 !== (int) $matched_recipe_id && (int) $recipe_id !== (int) $matched_recipe_id ) {
			return [
				'result' => false,
				'error'  => __( 'Recipe not matched.', 'uncanny-automator' ),
			];
		} elseif ( (int) $recipe_id === (int) $matched_recipe_id ) {
			if ( ! key_exists( $trigger_meta, $trigger['meta'] ) ) {
				return [
					'result' => false,
					'error'  => __( 'Trigger is completed.', 'uncanny-automator' ),
				];
			}
		}

		return $this->maybe_get_trigger_id( $user_id, $trigger_id, $recipe_id, $recipe_log_id );
	}

	/**
	 *
	 * Record an entry in to DB against a trigger
	 *
	 * @param      $user_id
	 * @param      $trigger_id
	 * @param      $recipe_id
	 * @param null $recipe_log_id
	 *
	 * @return array
	 */
	public function maybe_get_trigger_id( $user_id, $trigger_id, $recipe_id, $recipe_log_id = null ) {
		if ( null === $trigger_id || null === $recipe_id || null === $user_id ) {
			return [
				'result' => false,
				'error'  => __( 'One of the required field is missing.', 'uncanny-automator' ),
			];
		}

		$get_trigger_id = $this->get->trigger_log_id( $user_id, $trigger_id, $recipe_id, $recipe_log_id );

		if ( is_null( $get_trigger_id ) && is_numeric( $recipe_log_id ) ) {
			//Nothing found! Insert
			$get_trigger_id = $this->insert_trigger( $user_id, $trigger_id, $recipe_id, false, $recipe_log_id );
		}

		return [ 'result' => true, 'get_trigger_id' => $get_trigger_id, ];
	}

	/**
	 * Validate if the number of times of a trigger condition met
	 *
	 * @param $times_args
	 * @param bool $is_anonymous
	 *
	 * @return array
	 */
	public function maybe_trigger_num_times_completed( $times_args, $is_anonymous = false ) {
		$recipe_id      = key_exists( 'recipe_id', $times_args ) ? $times_args['recipe_id'] : null;
		$trigger_id     = key_exists( 'trigger_id', $times_args ) ? $times_args['trigger_id'] : null;
		$trigger        = key_exists( 'trigger', $times_args ) ? $times_args['trigger'] : null;
		$user_id        = key_exists( 'user_id', $times_args ) ? $times_args['user_id'] : null;
		$recipe_log_id  = key_exists( 'recipe_log_id', $times_args ) ? $times_args['recipe_log_id'] : null;
		$trigger_log_id = key_exists( 'trigger_log_id', $times_args ) ? $times_args['trigger_log_id'] : null;


		if ( $is_anonymous ) {
			$times_args['user_id']           = 0;
			$times_args['mark_as_anonymous'] = true;
		}

		if ( null === $trigger_id || null === $trigger || null === $user_id ) {
			return [
				'result' => false,
				'error'  => __( 'One of the required field is missing.', 'uncanny-automator' ),
			];
		}

		// The number of times the current user needs to visit the post/page
		$num_times  = key_exists( 'NUMTIMES', $trigger['meta'] ) ? absint( $trigger['meta']['NUMTIMES'] ) : 1;
		$run_number = $this->get->trigger_run_number( $trigger_id, $trigger_log_id, $user_id );

		// How many times has this user triggered this trigger
		if ( $is_anonymous ) {
			$user_num_times = null;
		} else {
			$user_num_times = $this->get->trigger_meta( $user_id, $trigger['ID'], 'NUMTIMES', $trigger_log_id );
		}
		$args = [
			'user_id'        => $user_id,
			'trigger_id'     => $trigger_id,
			'meta_key'       => 'NUMTIMES',
			'run_number'     => $run_number,
			'trigger_log_id' => $trigger_log_id,
		];

		if ( empty( $user_num_times ) ) {
			//This is first time user visited
			$args['meta_value'] = 1;
			$user_num_times     = 1;
		} else {

			$user_num_times ++;
			$args['run_number'] = $run_number + 1;
			$args['meta_value'] = 1;
		}

		$this->insert_trigger_meta( $args );
		//change completed from -1 to 0
		$this->maybe_change_recipe_log_to_zero( $recipe_id, $user_id, $recipe_log_id, true );

		// Move on if the user didn't trigger the trigger enough times
		if ( $user_num_times < $num_times ) {
			return [
				'result' => false,
				'error'  => __( 'Number of Times condition is not completed. ', 'uncanny-automator' ),
			];
		}

		// If the trigger was hit the enough times then complete the trigger
		if ( $user_num_times >= $num_times ) {
			return [
				'result'     => true,
				'error'      => __( 'Number of times condition met.', 'uncanny-automator' ),
				'run_number' => $args['run_number'],
			];
		}

		return [
			'result' => false,
			'error'  => __( 'Default Return. Something is wrong.', 'uncanny-automator' ),
		];
	}

	/**
	 * Validate if the number of times of a trigger condition met
	 *
	 * @param $option_meta
	 * @param null $save_for_option
	 *
	 * @return array
	 */
	public function maybe_trigger_add_any_option_meta( $option_meta, $save_for_option = null ) {
		if ( is_null( $save_for_option ) ) {
			return [
				'result' => false,
				'error'  => __( 'Option meta not defined.', 'uncanny-automator' ),
			];
		}

		$trigger_id     = key_exists( 'trigger_id', $option_meta ) ? $option_meta['trigger_id'] : null;
		$user_id        = key_exists( 'user_id', $option_meta ) ? $option_meta['user_id'] : null;
		$trigger_log_id = key_exists( 'trigger_log_id', $option_meta ) ? $option_meta['trigger_log_id'] : null;
		$post_id        = key_exists( 'post_id', $option_meta ) ? $option_meta['post_id'] : null;
		$is_signed_in   = key_exists( 'is_signed_in', $option_meta ) ? $option_meta['is_signed_in'] : false;
		$run_number     = $this->get->next_run_number( $option_meta['recipe_id'], $user_id, true );
		$trigger        = key_exists( 'trigger', $option_meta ) ? $option_meta['trigger'] : null;
		$trigger_meta   = ! empty( $save_for_option ) ? $save_for_option : null;


		if ( ! is_user_logged_in() && false === $is_signed_in ) {
			$option_meta['user_id']           = 0;
			$option_meta['mark_as_anonymous'] = true;
			/*return [
				'result' => false,
				'error'  => __( 'User is not logged in.', 'uncanny-automator' ),
			];*/
		}

		if ( null === $trigger_id || null === $trigger || null === $user_id ) {
			return [
				'result' => false,
				'error'  => __( 'One of the required field is missing.', 'uncanny-automator' ),
			];
		}

		$args = [
			'user_id'        => $user_id,
			'trigger_id'     => $trigger_id,
			'meta_key'       => $trigger_meta,
			'meta_value'     => $post_id,
			'run_number'     => $run_number,
			'trigger_log_id' => $trigger_log_id,
		];

		#Utilities::log( $args, '$meta_already_saved-$args', true, 'token-fixes' );
		$meta_already_saved = $this->get->maybe_get_meta_id_from_trigger_log( $run_number, $trigger_id, $trigger_log_id, $trigger_meta, $user_id );
		#Utilities::log( $meta_already_saved, '$meta_already_saved', true, 'token-fixes' );
		if ( ! $meta_already_saved ) {
			$this->insert_trigger_meta( $args );

			return [
				'result' => true,
				'error'  => __( 'Meta entry added.', 'uncanny-automator' ),
			];
		} elseif ( is_numeric( $meta_already_saved ) ) {
			$args['trigger_log_meta_id'] = $meta_already_saved;
			$this->update_trigger_meta( $user_id, $trigger_id, $trigger_meta, $post_id, $trigger_log_id );

			return [
				'result' => true,
				'error'  => __( 'Meta entry updated.', 'uncanny-automator' ),
			];
		}

		return [
			'result' => false,
			'error'  => __( 'No action happened.', 'uncanny-automator' ),
		];

	}
}
